/*
 * Copyright (C) 2018-2022 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Better handling for duplicate names
    2.2.5   Guard against occasional NPE on item delete removing base listeners
    2.9.6	When a Diary entry is added/changed, make sure updated comments are shown
    3.0.4	Comment handling
 */

package uk.co.gardennotebook.fxbean;

import javafx.beans.property.*;
import uk.co.gardennotebook.spi.*;
import uk.co.gardennotebook.util.StoryLineTree;
import java.util.Optional;
import java.util.List;
import java.util.ArrayList;
import java.beans.PropertyChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.beans.value.ChangeListener;

import javafx.collections.FXCollections;
import javafx.collections.ObservableList;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
//import org.apache.logging.log4j.Level;

import java.time.*;

/**
	*	Anywhere that items are purchased, includes garden centres, supermarkets, online shopping, etc, etc.
 *
 * @implNote
 * A more obvious name would be Supplier but that causes a potential name clash in Java
	*
	*	@author	Andy Gegg
	*	@version	3.0.4
	*	@since	1.0
*/
final public class RetailerBean implements INotebookBean
{
	private static final Logger LOGGER = LogManager.getLogger();

	private IRetailer baseItem = null;

	private Integer itemKey = 0;
	private boolean newItem = false;
	private boolean explicitSave = false;
	private final SimpleBooleanProperty saveRequiredProperty = new SimpleBooleanProperty(this, "saveRequired", explicitSave);
	private IRetailerBuilder explicitBuilder = null;

		// handle changes to the base item itself
	private PropertyChangeListener baseItemDeleted;
	private PropertyChangeListener baseItemReplaced;

	private final SimpleStringProperty nameProperty = new SimpleStringProperty(this, "name", "");
	private final ChangeListener<String> nameListener = this::onNameChange;
    private final ReadOnlyBooleanWrapper duplicateNameProperty = new ReadOnlyBooleanWrapper(this, "duplicateName", false);  //  2.2.0
	private final SimpleStringProperty descriptionProperty = new SimpleStringProperty(this, "description", "");
	private final ChangeListener<String> descriptionListener = this::onDescriptionChange;
	private final SimpleStringProperty addressProperty = new SimpleStringProperty(this, "address", "");
	private final ChangeListener<String> addressListener = this::onAddressChange;
	private final SimpleStringProperty webSiteProperty = new SimpleStringProperty(this, "webSite", "");
	private final ChangeListener<String> webSiteListener = this::onWebSiteChange;
	private final SimpleStringProperty eMailProperty = new SimpleStringProperty(this, "eMail", "");
	private final ChangeListener<String> eMailListener = this::onEMailChange;
	private final SimpleStringProperty phoneProperty = new SimpleStringProperty(this, "phone", "");
	private final ChangeListener<String> phoneListener = this::onPhoneChange;
	private final SimpleStringProperty mobileProperty = new SimpleStringProperty(this, "mobile", "");
	private final ChangeListener<String> mobileListener = this::onMobileChange;

	/*
	*	If true, this retailer ONLY supplies Products of their own brand
	*/
	private final SimpleBooleanProperty ownBrandOnlyProperty = new SimpleBooleanProperty(this, "ownBrandOnly", false);
	private final ChangeListener<Boolean> ownBrandOnlyListener = this::onOwnBrandOnlyChange;
	private final ReadOnlyObjectWrapper<LocalDateTime> lastUpdatedProperty = new ReadOnlyObjectWrapper<>(this, "lastUpdated", LocalDateTime.now());
	private final ReadOnlyObjectWrapper<LocalDateTime> createdProperty = new ReadOnlyObjectWrapper<>(this, "created", LocalDateTime.now());
	private ReadOnlyBooleanWrapper canDeleteProperty = null;
	private final ReadOnlyBooleanWrapper hasAncestorProperty = new ReadOnlyBooleanWrapper(this, "hasAncestor", false);
	private final ReadOnlyBooleanWrapper hasDescendantProperty = new ReadOnlyBooleanWrapper(this, "hasDescendant", false);

	private ReadOnlyBooleanWrapper isNewProperty = new ReadOnlyBooleanWrapper(this, "isNew", newItem);	//	2.9.6

	private BeanCommentHandler<IRetailer> beanCommentHandler;	//	2.9.6
	private final ReadOnlyStringWrapper commentTextProperty = new ReadOnlyStringWrapper(this, "commentText", "");

	private ObservableList<ProductBrandBean> childrenProductBrand = null;
	private PropertyChangeListener baseItemProductBrandChanged;

	private ObservableList<PurchaseBean> childrenPurchase = null;
	private PropertyChangeListener baseItemPurchaseChanged;

	private ObservableList<RetailerHasProductBean> childrenRetailerHasProduct = null;
	private PropertyChangeListener baseItemRetailerHasProductChanged;

	/**
	*	Construct an 'empty' Bean.  Set the various property values then call save() to create the new RetailerBean
	*/
	public RetailerBean()
	{
		this(null);
	}
	/**
	*	Construct a Bean wrapping the given Retailer
	*	If the parameter is null a new 'empty' Bean will be constructed
	*
	*	@param	initialValue	the Retailer to wrap.  If null an 'empty' bean will be constructed
	*
	*/
	public RetailerBean(final IRetailer initialValue)
	{
		ChangeListener<Boolean> saveRequiredListener = (obs, old, nval) -> {
			if (nval && !explicitSave)
			{
				explicitSave = true;
				ITrug server = TrugServer.getTrugServer().getTrug();
				explicitBuilder = server.getRetailerBuilder(baseItem);
			}
			if (!nval && explicitSave && (baseItem != null))
			{
				explicitSave = false;
				explicitBuilder = null;
			}
		};

		saveRequiredProperty.addListener(saveRequiredListener);

		if(initialValue == null)
		{
			newItem = true;
			//	add the listeners BEFORE setting values, or default values never get sent to the builder!
			addListeners();
			setDefaults();
			saveRequiredProperty.set(true);
			return;
		}

		baseItem = initialValue;

		itemKey = baseItem.getKey();

		newItem = false;
		setValues();

		addListeners();
		declareBaseListeners();
		addBaseListeners();
	}

	/**
	*	Returns all Retailer items wrapped as RetailerBean.
	*
	*	@return	a collection of RetailerBean beans
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public static ObservableList<RetailerBean> fetchAll() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetchAll()");
		ITrug server = TrugServer.getTrugServer().getTrug();
		IRetailerLister gal = server.getRetailerLister();
		List<RetailerBean> ll = gal.fetch().stream()
					.collect(ArrayList::new, (c, e) -> c.add(new RetailerBean(e)), ArrayList::addAll);
		LOGGER.traceExit();
		return FXCollections.observableArrayList(ll);
	}

	/**
	*	Returns the underlying Retailer, if present
	*
	*	@return	the underlying Retailer, if present
	*/
	public Optional<IRetailer> get()
	{
		return getValue();
	}

	/**
	*	Returns the underlying Retailer if present
	*
	*	@return	the underlying Retailer, if present
	*/
	public Optional<IRetailer> getValue()
	{
		return Optional.ofNullable(baseItem);
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.RETAILER;
	}

	@Override
	public Integer getKey()
	{
		return itemKey;
	}

	@Override
	public boolean sameAs(final INotebookBean other)
	{
		if (other == null || ((RetailerBean)other).baseItem == null || baseItem == null)
		{
			return false;
		}
		if (other.getType() != NotebookEntryType.RETAILER)
		{
			return false;
		}
		return baseItem.sameAs(((RetailerBean)other).baseItem);
	}

	@Override
	public boolean isNew()
	{
		return isNewProperty().get();
	}

	@Override
	public ReadOnlyBooleanProperty isNewProperty()
	{
		if (isNewProperty == null)
		{
			isNewProperty = new ReadOnlyBooleanWrapper(this, "isNew", newItem);
		}
		return isNewProperty.getReadOnlyProperty();
	}

	@Override
	public boolean canDelete() throws GNDBException
	{
		return canDeleteProperty().get();
	}

	@Override
	public ReadOnlyBooleanProperty canDeleteProperty() throws GNDBException
	{
		if (canDeleteProperty == null)
		{
			ITrug server = TrugServer.getTrugServer().getTrug();
			boolean canDel = server.getRetailerBuilder(baseItem).canDelete();
			canDeleteProperty = new ReadOnlyBooleanWrapper(this, "canDelete", canDel);
		}
		return canDeleteProperty.getReadOnlyProperty();
	}

	@Override
	public boolean hasAncestor() throws GNDBException
	{
		//	Retailer items do not participate in story lines
		return false;
	}

	@Override
	public ReadOnlyBooleanProperty hasAncestorProperty() throws GNDBException
	{
		//	Retailer items do not participate in story lines
		return hasAncestorProperty.getReadOnlyProperty();
	}	//	hasAncestorProperty()

	@Override
	public StoryLineTree<? extends INotebookBean> getAncestors() throws GNDBException
	{
		//	Retailer items do not participate in story lines
			return StoryLineTree.emptyTree();
	}	//	getAncestors()

	@Override
	public boolean hasDescendant() throws GNDBException
	{
		//	Retailer items do not participate in story lines
		return false;
	}

	@Override
	public ReadOnlyBooleanProperty hasDescendantProperty() throws GNDBException
	{
		//	Retailer items do not participate in story lines
		return hasDescendantProperty.getReadOnlyProperty();
	}	//	hasDescendantProperty()

	@Override
	public StoryLineTree<? extends INotebookBean> getDescendants() throws GNDBException
	{
		//	Retailer items do not participate in story lines
			return StoryLineTree.emptyTree();
	}	//	getDescendants()

	public String getName()
	{
		return nameProperty.get();
	}
	public void setName(final String newVal)
	{
		nameProperty.set(newVal);
	}
	/**
	*	Wraps the Name value of the Retailer
	*
	*	@return	a writable property wrapping the name attribute
	*/
	public StringProperty nameProperty()
	{
		return nameProperty;
	}

	private void onNameChange(ObservableValue<? extends String> obs, String old, String nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onNameChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onNameChange(): explicitSave");
			try
			{
				explicitBuilder.name(nval);
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		else
		{
LOGGER.debug("onNameChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getRetailerBuilder(baseItem).name(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

    /**
     * Use this to check if the name being applied is a duplicate of an existing name.
     * NB relies on calling {@code checkForDuplicateName} on focusLost.
     * 
     * @return a read-only indication that the name being set is a duplicate of an existing name
     * 
     * @since 2.2.0
     */
    public ReadOnlyBooleanProperty duplicateNameProperty()
    {
        return duplicateNameProperty.getReadOnlyProperty();
    }

    /**
     *  Must be called from the editor when the user has finished entering a new name value,
     * typically on a lost focus event.
     * 
     * @param newVal    the name the user is attempting to give
     * @return  true if newVal duplicates an existing name
     * 
     * @since 2.2.0
     */    
    public boolean checkForDuplicateName(final String newVal)
    {
        boolean duplicate = false;
    
		if (explicitSave)
		{
LOGGER.debug("checkNameDuplicate(): explicitSave");
			try
			{
				duplicate = explicitBuilder.isNameDuplicate(newVal);
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		else
		{
LOGGER.debug("checkNameDuplicate(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				duplicate = server.getRetailerBuilder(baseItem).isNameDuplicate(newVal);
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
        }
        duplicateNameProperty.set(duplicate);
        return duplicate;
    }

	public String getDescription()
	{
		return descriptionProperty.get();
	}
	public void setDescription(final String newVal)
	{
		descriptionProperty.set(newVal);
	}
	/**
	*	Wraps the Description value of the Retailer
	*
	*	@return	a writable property wrapping the description attribute
	*/
	public StringProperty descriptionProperty()
	{
		return descriptionProperty;
	}

	private void onDescriptionChange(ObservableValue<? extends String> obs, String old, String nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onDescriptionChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onDescriptionChange(): explicitSave");
			explicitBuilder.description(nval);
		}
		else
		{
LOGGER.debug("onDescriptionChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getRetailerBuilder(baseItem).description(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	public String getAddress()
	{
		return addressProperty.get();
	}
	public void setAddress(final String newVal)
	{
		addressProperty.set(newVal);
	}
	/**
	*	Wraps the Address value of the Retailer
	*
	*	@return	a writable property wrapping the address attribute
	*/
	public StringProperty addressProperty()
	{
		return addressProperty;
	}

	private void onAddressChange(ObservableValue<? extends String> obs, String old, String nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onAddressChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onAddressChange(): explicitSave");
			explicitBuilder.address(nval);
		}
		else
		{
LOGGER.debug("onAddressChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getRetailerBuilder(baseItem).address(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	public String getWebSite()
	{
		return webSiteProperty.get();
	}
	public void setWebSite(final String newVal)
	{
		webSiteProperty.set(newVal);
	}
	/**
	*	Wraps the WebSite value of the Retailer
	*
	*	@return	a writable property wrapping the webSite attribute
	*/
	public StringProperty webSiteProperty()
	{
		return webSiteProperty;
	}

	private void onWebSiteChange(ObservableValue<? extends String> obs, String old, String nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onWebSiteChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onWebSiteChange(): explicitSave");
			explicitBuilder.webSite(nval);
		}
		else
		{
LOGGER.debug("onWebSiteChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getRetailerBuilder(baseItem).webSite(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	public String getEMail()
	{
		return eMailProperty.get();
	}
	public void setEMail(final String newVal)
	{
		eMailProperty.set(newVal);
	}
	/**
	*	Wraps the EMail value of the Retailer
	*
	*	@return	a writable property wrapping the eMail attribute
	*/
	public StringProperty eMailProperty()
	{
		return eMailProperty;
	}

	private void onEMailChange(ObservableValue<? extends String> obs, String old, String nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onEMailChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onEMailChange(): explicitSave");
			explicitBuilder.eMail(nval);
		}
		else
		{
LOGGER.debug("onEMailChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getRetailerBuilder(baseItem).eMail(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	public String getPhone()
	{
		return phoneProperty.get();
	}
	public void setPhone(final String newVal)
	{
		phoneProperty.set(newVal);
	}
	/**
	*	Wraps the Phone value of the Retailer
	*
	*	@return	a writable property wrapping the phone attribute
	*/
	public StringProperty phoneProperty()
	{
		return phoneProperty;
	}

	private void onPhoneChange(ObservableValue<? extends String> obs, String old, String nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onPhoneChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onPhoneChange(): explicitSave");
			explicitBuilder.phone(nval);
		}
		else
		{
LOGGER.debug("onPhoneChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getRetailerBuilder(baseItem).phone(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	public String getMobile()
	{
		return mobileProperty.get();
	}
	public void setMobile(final String newVal)
	{
		mobileProperty.set(newVal);
	}
	/**
	*	Wraps the Mobile value of the Retailer
	*
	*	@return	a writable property wrapping the mobile attribute
	*/
	public StringProperty mobileProperty()
	{
		return mobileProperty;
	}

	private void onMobileChange(ObservableValue<? extends String> obs, String old, String nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onMobileChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onMobileChange(): explicitSave");
			explicitBuilder.mobile(nval);
		}
		else
		{
LOGGER.debug("onMobileChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getRetailerBuilder(baseItem).mobile(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	public boolean isOwnBrandOnly()
	{
		return ownBrandOnlyProperty.get();
	}
	public void setOwnBrandOnly(final boolean newVal)
	{
		ownBrandOnlyProperty.set(newVal);
	}
	/**
	*	Wraps the OwnBrandOnly value of the Retailer
	*
	*	@return	a writable property wrapping the ownBrandOnly attribute
	*/
	public BooleanProperty ownBrandOnlyProperty()
	{
		return ownBrandOnlyProperty;
	}

	private void onOwnBrandOnlyChange(ObservableValue<? extends Boolean> obs, boolean old, boolean nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onOwnBrandOnlyChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onOwnBrandOnlyChange(): explicitSave");
			explicitBuilder.ownBrandOnly(nval);
		}
		else
		{
LOGGER.debug("onOwnBrandOnlyChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getRetailerBuilder(baseItem).ownBrandOnly(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	public LocalDateTime getLastUpdated()
	{
		return lastUpdatedProperty.get();
	}
	/**
	*	Wraps the LastUpdated value of the Retailer
	*	Note that this value cannot be changed by the user
	*
	*	@return	a read-only property wrapping the lastUpdated attribute
	*/
	public ReadOnlyObjectProperty<LocalDateTime> lastUpdatedProperty()
	{
		return lastUpdatedProperty.getReadOnlyProperty();
	}

	public LocalDateTime getCreated()
	{
		return createdProperty.get();
	}
	/**
	*	Wraps the Created value of the Retailer
	*	Note that this value cannot be changed by the user
	*
	*	@return	a read-only property wrapping the created attribute
	*/
	public ReadOnlyObjectProperty<LocalDateTime> createdProperty()
	{
		return createdProperty.getReadOnlyProperty();
	}

	/**
	*	Return a list of any ProductBrand of this Retailer or an empty list
	*
	*	@return	A list of ProductBrand items, possibly empty
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public ObservableList<ProductBrandBean> getProductBrand() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("getProductBrand()");
		if (childrenProductBrand == null)
		{
			childrenProductBrand = FXCollections.observableArrayList();
			ITrug server = TrugServer.getTrugServer().getTrug();
			for (IProductBrand ix : server.getProductBrandLister().retailer(baseItem).fetch())
			{
				childrenProductBrand.add(new ProductBrandBean(ix));
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
		return childrenProductBrand;
	}

	/**
	*	Return a list of any Purchase of this Retailer or an empty list
	*
	*	@return	A list of Purchase items, possibly empty
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public ObservableList<PurchaseBean> getPurchase() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("getPurchase()");
		if (childrenPurchase == null)
		{
			childrenPurchase = FXCollections.observableArrayList();
			ITrug server = TrugServer.getTrugServer().getTrug();
			for (IPurchase ix : server.getPurchaseLister().retailer(baseItem).fetch())
			{
				childrenPurchase.add(new PurchaseBean(ix));
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
		return childrenPurchase;
	}

	/**
	*	Return a list of any RetailerHasProduct of this Retailer or an empty list
	*
	*	@return	A list of RetailerHasProduct items, possibly empty
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public ObservableList<RetailerHasProductBean> getRetailerHasProduct() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("getRetailerHasProduct()");
		if (childrenRetailerHasProduct == null)
		{
			childrenRetailerHasProduct = FXCollections.observableArrayList();
			ITrug server = TrugServer.getTrugServer().getTrug();
			for (IRetailerHasProduct ix : server.getRetailerHasProductLister().retailer(baseItem).fetch())
			{
				childrenRetailerHasProduct.add(new RetailerHasProductBean(ix));
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
		return childrenRetailerHasProduct;
	}

	@Override
	public ObservableList<CommentBean> getComments()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("getComments()");
		return LOGGER.traceExit(log4jEntryMsg, beanCommentHandler.getComments());
	}	//	getComments()

	//	2.9.6
	@Override
	public ReadOnlyStringProperty commentTextProperty()
	{
//		return beanCommentHandler.commentTextProperty();
		commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
		return commentTextProperty;
	}

	@Override
	public void addComment(final String text) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment({})", text);
		if (text == null || text.isBlank()) return;	//	2.9.6

		beanCommentHandler.addComment(text);	//	2.9.6

		if (explicitSave)
		{
LOGGER.debug("addComment(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent Retailer
LOGGER.debug("addComment(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getRetailerBuilder(baseItem).addComment(text).save();
			setValues();	//	2.9.6
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	addComment()

	//	2.9.6
	@Override
	public void addComment(CommentBean comment) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment(comment bean): comment: {}, text : {}", comment, comment==null ? "null" :comment.getComment());
		if (comment == null) return;
		if (comment.getParentType() != this.getType()) return;
		if (comment.getComment() == null || comment.getComment().isBlank()) return;

		beanCommentHandler.addComment(comment);

		if (explicitSave)
		{
			LOGGER.debug("addComment(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent Retailer
			LOGGER.debug("addComment(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getRetailerBuilder(baseItem).addComment(comment.getComment()).save();
			setValues();	//	2.9.6
		}
		LOGGER.debug("addComment(comment bean): commentTextProperty: {}", commentTextProperty().get());

		LOGGER.traceExit(log4jEntryMsg);
	}

	@Override
	public void changeCommentText(final CommentBean comment, final String text) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeCommentText(): comment={}, text={}", comment, text);
		if (text == null || text.isBlank()) return;

		if (comment == null)	//	2.9.6
		{
			addComment(text);
			return;
		}

		if (comment.getParentType() != this.getType()) return;

		beanCommentHandler.changeCommentText(comment, text);

		if (explicitSave)
		{
LOGGER.debug("changeCommentText(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent Retailer
LOGGER.debug("changeCommentText(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getRetailerBuilder(baseItem).changeComment(comment.get().get(), text).save();
			setValues();	//	2.9.6
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	changeCommentText()

	@Override
	public void changeCommentDate(CommentBean comment, final LocalDate date) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeCommentDate(): comment={}, date={}", comment, date);
		if (date == null) return;

		//	2.9.6
		if (comment == null)
		{
			return;
		}

		if (comment.getParentType() != this.getType()) return;

		beanCommentHandler.changeCommentDate(comment, date);

		if (explicitSave)
		{
LOGGER.debug("changeCommentDate(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent Retailer
LOGGER.debug("changeCommentDate(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getRetailerBuilder(baseItem).changeComment(comment.get().get(), date).save();
			setValues();	//	2.9.6
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	changeCommentDate()

	@Override
	public void deleteComment(CommentBean comment) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment(): comment={}", comment);
		if (comment == null) return;

		if (comment.getParentType() != this.getType()) return;

		beanCommentHandler.deleteComment(comment);

		if (explicitSave)
		{
LOGGER.debug("deleteComment(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent Retailer
LOGGER.debug("deleteComment(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getRetailerBuilder(baseItem).deleteComment(comment.get().get()).save();
			setValues();	//	2.9.6
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	deleteComment()

	public boolean isSaveRequired()
	{
		return explicitSave;
	}
	public void setSaveRequired(boolean reqd)
	{
		saveRequiredProperty.set(reqd);
	}
	public BooleanProperty saveRequiredProperty()
	{
		return saveRequiredProperty;
	}

	public boolean needSave()
	{
		if (!explicitSave)
			return false;

		return explicitBuilder.needSave() || beanCommentHandler.needSave();
	}

	public boolean canSave()
	{
		if (!explicitSave)
			return true;

		return explicitBuilder.canSave();
	}

	/**
	*	Save changes to the underlying Retailer item
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public void save() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("save()");
		if (!explicitSave) return;

		saveComments();	//	2.9.6 - do this here so that explicitBuilder knows there's a change

		if (!explicitBuilder.needSave())
		{
			return;
		}
		if (!explicitBuilder.canSave())
		{
			throw new IllegalStateException("RetailerBean: cannot save at this time - mandatory values not set");
		}

		baseItem = explicitBuilder.save();
		LOGGER.debug("save(): after explicitBuilder.save(): comments: {}", ()->baseItem.getComments());	//	2.9.6
		setValues();	//	2.9.6
		saveRequiredProperty.set(false);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	save()

	//	2.9.6
	private void saveComments()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("saveComments()");

		beanCommentHandler.saveComments(
				cb -> explicitBuilder.addComment(cb.getComment()),	//	add
				cb -> explicitBuilder.changeComment(cb.get().get(), cb.getComment()),	//	change text
				cb -> explicitBuilder.changeComment(cb.get().get(), cb.getDate()),	//	change date
				cb -> explicitBuilder.deleteComment(cb.get().get())		//	delete
		);
	}

	/**
	*	Delete the underlying Retailer item
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public void delete() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("delete()");
		if (newItem) return;

		if (explicitSave)
		{
			explicitBuilder.delete();
			saveRequiredProperty.set(false);
		}
		else
		{
			ITrug server = TrugServer.getTrugServer().getTrug();
			server.getRetailerBuilder(baseItem).delete();
		}
	}	//	delete()

	public void cancelEdit()
	{
		if (!explicitSave) return;
		if (newItem) return;

		saveRequiredProperty.set(false);
		setValues();
	}

	private void setDefaults()
	{
		saveRequiredProperty.setValue(false);
		nameProperty.setValue("");
		descriptionProperty.setValue("");
		addressProperty.setValue("");
		webSiteProperty.setValue("");
		eMailProperty.setValue("");
		phoneProperty.setValue("");
		mobileProperty.setValue("");
		ownBrandOnlyProperty.setValue(false);
		lastUpdatedProperty.setValue(LocalDateTime.now());
		createdProperty.setValue(LocalDateTime.now());
		childrenProductBrand = null;
		baseItemProductBrandChanged = null;
		childrenPurchase = null;
		baseItemPurchaseChanged = null;
		childrenRetailerHasProduct = null;
		baseItemRetailerHasProductChanged = null;

		//	2.9.6
		isNewProperty.set(true);
		beanCommentHandler = new BeanCommentHandler<>(this, baseItem);
		commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
	}

	private void setValues()
	{
		saveRequiredProperty.setValue(false);
		nameProperty.setValue(baseItem.getName());
		descriptionProperty.setValue(baseItem.getDescription().orElse(""));
		addressProperty.setValue(baseItem.getAddress().orElse(""));
		webSiteProperty.setValue(baseItem.getWebSite().orElse(""));
		eMailProperty.setValue(baseItem.getEMail().orElse(""));
		phoneProperty.setValue(baseItem.getPhone().orElse(""));
		mobileProperty.setValue(baseItem.getMobile().orElse(""));
		ownBrandOnlyProperty.setValue(baseItem.isOwnBrandOnly());
		lastUpdatedProperty.setValue(baseItem.getLastUpdated());
		createdProperty.setValue(baseItem.getCreated());

		itemKey = baseItem.getKey();
		newItem = false;
		isNewProperty.set(false);	//	2.9.6

		LOGGER.debug("setvalues(): about to change BeanCommentHandler");
		beanCommentHandler = new BeanCommentHandler<>(this, baseItem);
		commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
	}

	private void addListeners()
	{
		nameProperty.addListener(nameListener);
		descriptionProperty.addListener(descriptionListener);
		addressProperty.addListener(addressListener);
		webSiteProperty.addListener(webSiteListener);
		eMailProperty.addListener(eMailListener);
		phoneProperty.addListener(phoneListener);
		mobileProperty.addListener(mobileListener);
		ownBrandOnlyProperty.addListener(ownBrandOnlyListener);
	}
	private void removeListeners()
	{
		nameProperty.removeListener(nameListener);
		descriptionProperty.removeListener(descriptionListener);
		addressProperty.removeListener(addressListener);
		webSiteProperty.removeListener(webSiteListener);
		eMailProperty.removeListener(eMailListener);
		phoneProperty.removeListener(phoneListener);
		mobileProperty.removeListener(mobileListener);
		ownBrandOnlyProperty.removeListener(ownBrandOnlyListener);
	}
	private void declareBaseListeners()
	{
		// handle changes to the base item itself
		baseItemDeleted = evt -> {
				removeListeners();
				removeBaseListeners();
				setDefaults();
				baseItem = null;
			};
		baseItemReplaced = evt -> {
				if (evt.getNewValue() != null)
				{
					removeBaseListeners();
					baseItem = (IRetailer)(evt.getNewValue());
					setValues();
					addBaseListeners();
				}
			};

		baseItemProductBrandChanged = evt -> {
				if (childrenProductBrand == null)
				{
					return;
				}
				if (evt.getNewValue() != null)
				{
					if (!(evt.getNewValue() instanceof IProductBrand))
					{
						throw new IllegalArgumentException("baseItemProductBrandChanged: newVal wrong type");
					}
					childrenProductBrand.add(new ProductBrandBean((IProductBrand)(evt.getNewValue())));
				}
				else if (evt.getOldValue() != null)
				{
					if (!(evt.getOldValue() instanceof IProductBrand))
					{
						throw new IllegalArgumentException("baseItemProductBrandChanged: oldVal wrong type");
					}
					//	When the db item is deleted it fires an event which is picked up here AND in the child bean
					//	The child bean sets its underlying baseItem to null so getValue() returns an Optional of null
					//	The order in which the event handlers are called is unpredictable
					childrenProductBrand.removeIf(pr -> (pr.getValue().isEmpty()) ||
						(pr.getValue().
							get().
							getKey().
							equals(((IProductBrand)(evt.getOldValue())).getKey())));
				}
			};

		baseItemPurchaseChanged = evt -> {
				if (childrenPurchase == null)
				{
					return;
				}
				if (evt.getNewValue() != null)
				{
					if (!(evt.getNewValue() instanceof IPurchase))
					{
						throw new IllegalArgumentException("baseItemPurchaseChanged: newVal wrong type");
					}
					childrenPurchase.add(new PurchaseBean((IPurchase)(evt.getNewValue())));
				}
				else if (evt.getOldValue() != null)
				{
					if (!(evt.getOldValue() instanceof IPurchase))
					{
						throw new IllegalArgumentException("baseItemPurchaseChanged: oldVal wrong type");
					}
					//	When the db item is deleted it fires an event which is picked up here AND in the child bean
					//	The child bean sets its underlying baseItem to null so getValue() returns an Optional of null
					//	The order in which the event handlers are called is unpredictable
					childrenPurchase.removeIf(pr -> (pr.getValue().isEmpty()) ||
						(pr.getValue().
							get().
							getKey().
							equals(((IPurchase)(evt.getOldValue())).getKey())));
				}
			};

		baseItemRetailerHasProductChanged = evt -> {
				if (childrenRetailerHasProduct == null)
				{
					return;
				}
				if (evt.getNewValue() != null)
				{
					if (!(evt.getNewValue() instanceof IRetailerHasProduct))
					{
						throw new IllegalArgumentException("baseItemRetailerHasProductChanged: newVal wrong type");
					}
					childrenRetailerHasProduct.add(new RetailerHasProductBean((IRetailerHasProduct)(evt.getNewValue())));
				}
				else if (evt.getOldValue() != null)
				{
					if (!(evt.getOldValue() instanceof IRetailerHasProduct))
					{
						throw new IllegalArgumentException("baseItemRetailerHasProductChanged: oldVal wrong type");
					}
					//	When the db item is deleted it fires an event which is picked up here AND in the child bean
					//	The child bean sets its underlying baseItem to null so getValue() returns an Optional of null
					//	The order in which the event handlers are called is unpredictable
					childrenRetailerHasProduct.removeIf(pr -> (pr.getValue().isEmpty()) ||
						(pr.getValue().
							get().
							getKey().
							equals(((IRetailerHasProduct)(evt.getOldValue())).getKey())));
				}
			};

	}
    
	private void addBaseListeners()
	{
        if (baseItem == null) return;
		baseItem.addPropertyChangeListener("deleted", baseItemDeleted);
		baseItem.addPropertyChangeListener("replaced", baseItemReplaced);

		baseItem.addPropertyChangeListener("ProductBrand", baseItemProductBrandChanged);

		baseItem.addPropertyChangeListener("Purchase", baseItemPurchaseChanged);

		baseItem.addPropertyChangeListener("RetailerHasProduct", baseItemRetailerHasProductChanged);

	}
	private void removeBaseListeners()
	{
        if (baseItem == null) return;
		baseItem.removePropertyChangeListener("deleted", baseItemDeleted);
		baseItem.removePropertyChangeListener("replaced", baseItemReplaced);

		baseItem.removePropertyChangeListener("ProductBrand", baseItemProductBrandChanged);

		baseItem.removePropertyChangeListener("Purchase", baseItemPurchaseChanged);

		baseItem.removePropertyChangeListener("RetailerHasProduct", baseItemRetailerHasProductChanged);

	}

	@Override
	public String toString()
	{
		return "RetailerBean wrapping " + baseItem;
	}

}

