/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.LocationBean;
import uk.co.gardennotebook.fxbean.WeatherConditionBean;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.ILocation;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 *	Physical or conceptual areas of the garden.
 *
 *	Must be public to be loaded by FXML
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	3.0.0
 */
public final class LocationCombo extends DiaryEntryCombo<LocationBean>
{
    private static final Logger LOGGER = LogManager.getLogger();

    public LocationCombo()
    {
        super();
        try
        {
            this.getItems().setAll(LocationBean.fetchAll());
        } catch (GNDBException ex) {
            PanicHandler.panic(ex);
        }
        super.currentValue = null;
    }

    public LocationCombo(LocationBean initialVal)
    {
        this();
        this.setValue(initialVal);
        super.currentValue = initialVal;
    }

    void excludeLocations(List<? extends LocationBean> exclusions)
    {
        LOGGER.debug("excludeLocations(): exclusions: {}", exclusions);
        List<Integer> excludeKeys = exclusions.stream().map(t -> t.getKey()).collect(Collectors.toList());
        this.getItems().removeIf(t -> excludeKeys.contains( t.getKey()));
    }

    void excludeLocations(LocationBean... exclusions)
    {
        LOGGER.debug("excludeLocations(): exclusions: {}", exclusions);
        excludeLocations(Arrays.asList(exclusions));
    }

    @Override
    String getName(LocationBean item)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("getName({})", item);
        if (item == null)
        {
            return "";
        }
        if (item instanceof LocationBean)
        {
            return ((LocationBean)item).getName();
        }
        else
        {
            throw new UnsupportedOperationException("LocationCombo: getName: unknown object in converter: "+item);
        }
    }	//	getName()

    @Override
    LocationBean newItem(String name)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("newItem({})", name);
        if (name == null || name.isEmpty())
        {
            return null;
        }
        LocationBean added = new LocationBean();
        added.setName(name);
        try
        {
            added.save();
        } catch (GNDBException ex) {
            PanicHandler.panic(ex);
        }
        return LOGGER.traceExit(log4jEntryMsg,added);
    }	//	newItem()
}
