/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 
 *
 *   This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software:  you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Allow date choice to be restricted to after today
	3.0.1	Check for null value in DatePicker
 */

package uk.co.gardennotebook;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;

import javafx.scene.control.DateCell;
import javafx.scene.control.DatePicker;
import javafx.scene.layout.VBox;
import javafx.util.Callback;
import uk.co.gardennotebook.fxbean.INotebookBean;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 *	Simplifies table cell editing where a date picker is required.
 *
 * @param <S> the type of the TableView
 * 
 * @author Andy Gegg
 * @version 3.0.1
 * @since 1.1.0
 */
class NotebookDateCellType<S extends INotebookBean> extends NotebookEditCellType<S, LocalDate>
{
	private static final Logger LOGGER = LogManager.getLogger();

	final boolean allowBeforeToday;
/**
 * Allow dates to be chosen before the current date.
 */	
	NotebookDateCellType()
	{
		this(true);
	}

	/**
	 * Specify whether dates prior to today's date are permitted.
	 *
	 * @param allowBeforeToday	if true, all dates are allowed, if false only dates on or after today are permitted.
	 */
	NotebookDateCellType(boolean allowBeforeToday)
	{
		this.allowBeforeToday = allowBeforeToday;
	}
	
	@Override
	void updateViewMode(S rowItem, LocalDate cellItem) {
		LOGGER.debug("NotebookDateCellType: updateViewMode(): item: {}", cellItem);
		LOGGER.debug("NotebookDateCellType: updateViewMode(): rowItem: {}", rowItem);
		setGraphic(null);
		setText(null);
		
		if (isEditing() && mayEdit(rowItem, cellItem))
		{
			VBox vBox = new VBox();
			DatePicker dp = new DatePicker((cellItem == null) ? LocalDate.now() : cellItem);
			dp.setEditable(true);
			dp.setOnAction(ev -> {
				LOGGER.debug("NotebookDateCellType: setOnAction: selection: {}", dp.getValue());
//				if (!allowBeforeToday && dp.getValue().isBefore(LocalDate.now()))
				if (!allowBeforeToday && dp.getValue() != null && dp.getValue().isBefore(LocalDate.now()))	//	3.0.1
				{
					LOGGER.debug("about to cancelEdit");
					this.cancelEdit();
					return;
				}
				commitEdit(dp.getValue());
			});

			if (!allowBeforeToday)
			{
				dp.setDayCellFactory(new Callback<>()
				{
					public DateCell call(final DatePicker datePicker)
					{
						return new DateCell()
						{
							@Override
							public void updateItem(LocalDate item, boolean empty)
							{
								super.updateItem(item, empty);
								if (item.isBefore(LocalDate.now()))
								{
									setDisable(true);
								}
							}
						};
					}
				});

			}
			vBox.getChildren().add(dp);
			setGraphic(vBox);
		}
		else
		{
			if (cellItem != null) 
			{// can be null if cancelEdit was called, e.g. if user leaves the table cell
				setText(cellItem.format(DateTimeFormatter.ofLocalizedDate(FormatStyle.SHORT)));
			}
		}
		
	}
	
}
