/*
 * Copyright (C) 2018 Andrew Gegg
 
 *
 *   This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software:  you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    2.4.0   Added an easy way to check editablility
 */

package uk.co.gardennotebook;

import java.io.IOException;
import java.util.ResourceBundle;

import javafx.beans.property.ReadOnlyStringProperty;
import javafx.beans.property.ReadOnlyStringWrapper;
import javafx.fxml.FXML;
import javafx.fxml.FXMLLoader;
import javafx.scene.control.TextArea;
import javafx.scene.control.TextField;
import javafx.scene.layout.GridPane;
import uk.co.gardennotebook.fxbean.PlantNoteBean;
import uk.co.gardennotebook.fxbean.PlantSpeciesBean;
import uk.co.gardennotebook.fxbean.PlantVarietyBean;

/**
 * Controller class for the Plant Note tab.
 * These tabs are displayed in the PlantSpecies and PlantVariety editors
 *
	*	@author Andy Gegg
	*	@version	2.4.0
	*	@since	1.1.0
 */
final class PlantNoteTab extends GridPane
{

	private final PlantNoteBean thisValueBean;

	// if false, neither the title nor the text may be edited
	// this is to allow species notes to be shown on variety editors
	private boolean editable = true;

	private boolean addingItem;

	private final ReadOnlyStringWrapper titleProperty = new ReadOnlyStringWrapper(this, "noteTitle", null);

	@FXML
	private ResourceBundle resources;

	@FXML
	private TextField fldTitle;
	@FXML
	private TextArea fldNote;

	private PlantNoteTab()
	{
		this((PlantNoteBean)null);
	}
	
	PlantNoteTab(PlantNoteBean initialVal)
	{
		this(initialVal, true);
	}
	
	PlantNoteTab(PlantNoteBean initialVal, boolean editable)
	{
		this.thisValueBean = (initialVal != null ? initialVal : new PlantNoteBean());
		this.addingItem = (initialVal == null);
		this.editable = editable;

		FXMLLoader fxmlLoader = new FXMLLoader(getClass().getResource("/fxml/PlantNoteTab.fxml"),
			ResourceBundle.getBundle("notebook") );
		fxmlLoader.setRoot(this);
		fxmlLoader.setController(this);
		try {
			fxmlLoader.load();	// NB initialize is called from in here
		} catch (IOException exception) {
			throw new RuntimeException(exception);
		}		
	}
	
	PlantNoteTab(PlantSpeciesBean plantSpecies)
	{
		this();
        if (!plantSpecies.isNew())
        {
            this.thisValueBean.setPlantSpecies(plantSpecies);
        }
	}
	
	PlantNoteTab(PlantVarietyBean plantVariety)
	{
		this();
        if (!plantVariety.isNew())
        {
            this.thisValueBean.setPlantSpecies(plantVariety.getPlantSpecies());
            this.thisValueBean.setPlantVariety(plantVariety);
        }
	}
	
	/*
	 * Initializes the controller class.
	 */
	@FXML
	public void initialize()
	{
		fldTitle.textProperty().bindBidirectional(thisValueBean.titleProperty());
		fldTitle.setDisable(!editable);
		fldNote.textProperty().bindBidirectional(thisValueBean.noteProperty());
		fldNote.setDisable(!editable);
		titleProperty.bind(fldTitle.textProperty());		
	}
	
	ReadOnlyStringProperty titleProperty()
	{
		return titleProperty.getReadOnlyProperty();
	}
	
	PlantNoteBean getPlantNote()
	{
		return thisValueBean;
	}
    
    /**
     * Check if the Note is editable.  In particular, used to prevent the PlantVarietyEditor deleting Species notes
     * 
     * @return true if the Note is editable
     * 
     * @since 2.4.0
     */
    boolean isEditable()
    {
        return editable;
    }

}
