/*
 * Copyright (C) 2020 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    2.8.0   first version - replaces ProductDetailCombo<ProductBean>
 */

package uk.co.gardennotebook;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import uk.co.gardennotebook.fxbean.ProductBean;
import uk.co.gardennotebook.fxbean.ProductCategoryBean;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.property.SimpleStringProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import org.apache.logging.log4j.Level;
import uk.co.gardennotebook.fxbean.ProductBrandBean;
import uk.co.gardennotebook.spi.GNDBException;

/**
	*	Combo box for ProductNameDetail property on non-plantlike products.
	*
	*	Must be public to be loaded by FXML
	*
	*	@author	Andy Gegg
	*	@version	2.8.0
	*	@since	2.8.0
*/
final public class ProductDetailCombo extends DiaryEntryCombo<String>
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final SimpleObjectProperty<ProductCategoryBean> productCategory = new SimpleObjectProperty<>(this, "ParentProductCategory", null);
	private final SimpleStringProperty productName = new SimpleStringProperty(this, "ParentName", null);

//	need a no-args constructor for FXML Loader
	public ProductDetailCombo()
	{
		super();
		super.currentValue = null;
        productName.addListener((obs, oldVal, newVal) -> {
				if (newVal == null)
				{
					this.getItems().clear();
					this.setValue(null);
				}
				if (oldVal != null && oldVal.equals(productNameProperty().getValue()))
				{// not actually changed - can happen during initialisation
					return;
				}
				else
				{
                    List<ProductBean> prods = null;
					try {
						prods = ProductBean.fetchAll(productCategory.get());
					} catch (GNDBException ex) {
						PanicHandler.panic(ex);
					}
                    List<String> justNames = prods.stream()
                                                .filter(x -> x.getName().equalsIgnoreCase(newVal))
                                                .map(ProductBean::getNameDetail_1)
                                                .filter(x -> x != null && !x.isBlank()) // this.getValue() can be null, getDetail can be null
                                                .distinct() 
                                                .sorted(String::compareToIgnoreCase)
                                                .collect(Collectors.toList());
                    this.getItems().setAll(justNames);
				}
            
        });
	}

	public ProductDetailCombo(ProductCategoryBean parent)
	{
		this();
		productCategoryProperty().set(parent);
	}

	public ProductDetailCombo(ProductCategoryBean parent, ProductBean initialVal)
	{
		this(parent);
        productName.set(initialVal.getName());
		this.setValue(initialVal.getNameDetail_1());
		super.currentValue = initialVal.getNameDetail_1();
	}

	SimpleObjectProperty<ProductCategoryBean> productCategoryProperty()
	{
		return productCategory;
	}

	SimpleStringProperty productNameProperty()
	{
		return productName;
	}

	@Override
	String getName(String item)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("getName({})", item);
		if (item == null)
		{
			return "";
		}
        return item;
	}	//	getName()

	@Override
	String newItem(String name)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("newItem({})", name);
		if (name == null || name.isEmpty())
		{
			return null;
		}
		return LOGGER.traceExit(log4jEntryMsg, name);
	}	//	newItem()
}
