/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.spi;

import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.util.Optional;

/**
 * A more detailed plan of which species and, possibly, varieties will be planted in the referred CroppingPlan.
 * For instance, in 'Alliums' we might have 'leek' and 'shallot'.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	3.0.0
 */
public interface ICroppingActual extends INotebookEntry
{
    /**
     *	The owning CroppingPlan.
     *
     *	@return	the CroppingPlan
     */
    public ICroppingPlan getCroppingPlan();

    /**
     *	The owning PlantSpecies.
     *
     *	@return	the PlantSpecies
     */
    public IPlantSpecies getPlantSpecies();

    /**
     *	The owning PlantVariety.
     *
     *	@return	the PlantVariety
     */
    public Optional<IPlantVariety> getPlantVariety();

    /**
     *	Get the value of lastUpdated.
     *
     *	@return	the value of lastUpdated
     */
    public LocalDateTime getLastUpdated();

    /**
     *	Get the value of created.
     *
     *	@return	the value of created
     */
    public LocalDateTime getCreated();

    /**
     *	Add a change listener to the CroppingActual.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this Husbandry, newValue is an added comment, oldValue is a deleted comment
    </UL>
     *	@param	listener	a PropertyChangeListener (may be a Lambda expression)
     */
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Remove a change listener from the CroppingActual.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this Husbandry, newValue is an added comment, oldValue is a deleted comment
    </UL>
     *	@param	listener	the listener to remove
     */
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Signal to all listeners that this CroppingActual has been deleted.
     *
     */
    public void flagDeleted();

    /**
     *	Signal to all listeners that this CroppingActual has been replaced.
     *
     *	@param	newValue	the replacement CroppingActual
     */
    public void flagReplaced(ICroppingActual newValue);

}
