/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.spi;

import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.time.Year;
import java.util.Optional;

/**
 * This records the type of crop grown in a given location in a given planting year.  For instance, 'Alliums' in 'Bed 1'
 * in 2021.  Details of the actual plant species and, possibly, varieties can be added through the CroppingActual table.
 * Entries for past years are records of what was grown, entries for future years are to record the intended rotation of
 * crops through the various beds; the intention is that more detail is added (in CroppingActual) as plants are
 * ordered and sown.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	3.0.0
 */

public interface ICroppingPlan extends INotebookEntry
{
    /**
     * Get the CropRotationGroup in use in this Plan
     *
     * @return	the CropRotationGroup
     */
    public ICropRotationGroup getCropRotationGroup();

    /**
     *	The Location covered by this Plan.
     *
     *	@return	the Location
     */
    public ILocation getLocation();

    /**
     *	Get the Year covered by this Plan.
     *
     *	@return	the value of date
     */
    public Year getYearOfPlan();

    /**
     *	Get the value of lastUpdated.
     *
     *	@return	the value of lastUpdated
     */
    public LocalDateTime getLastUpdated();

    /**
     *	Get the value of created.
     *
     *	@return	the value of created
     */
    public LocalDateTime getCreated();

    /**
     * Get the planting details for this Plan
     *
     * @return  a list of (intended) crops
     */
    public ICroppingActualLister getCroppingActual();

    /**
     *	Add a change listener to the CroppingPlan.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this Husbandry, newValue is an added comment, oldValue is a deleted comment
     *			<LI>CroppingActual	-	this CroppingActual has been added to or deleted from this CroppingPlan's children, newValue is an added object, oldValue is the deleted object
    </UL>
     *	@param	listener	a PropertyChangeListener (may be a Lambda expression)
     */
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Remove a change listener from the CroppingPlan.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this Husbandry, newValue is an added comment, oldValue is a deleted comment
     *			<LI>CroppingActual	-	this CroppingActual has been added to or deleted from this CroppingPlan's children, newValue is an added object, oldValue is the deleted object
    </UL>
     *	@param	listener	the listener to remove
     */
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Signal to all listeners that this CroppingPlan has been deleted.
     *
     */
    public void flagDeleted();

    /**
     *	Signal to all listeners that this CroppingPlan has been replaced.
     *
     *	@param	newValue	the replacement CroppingPlan
     */
    public void flagReplaced(ICroppingPlan newValue);

    /**
     *	Signal to all listeners that a child has been deleted.
     *
     *	@param	child	this CroppingActual is no longer a child of this CroppingPlan
     */
    public void flagChildDeleted(ICroppingActual child);

    /**
     *	Signal to all listeners that a child has been added.
     *
     *	@param	child	this CroppingActual has been added as a child of this CroppingPlan
     */
    public void flagChildAdded(ICroppingActual child);

}
