/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Support converting ToDoList items (back) into Reminders to delay the action.
			Support 'watch-for' entries from PurchaseItems.
 */

package uk.co.gardennotebook.spi;

import java.beans.PropertyChangeListener;
import java.util.List;
import java.time.*;

import java.util.Optional;

/**
*	A list of reminders of things to do. <BR>Reminders can be 'one-off' or repeated at various intervals (e.g. 'sow lettuce in March' shown every year.).
*
*	@author	Andy Gegg
*	@version	3.0.0
*	@since	1.0
*/
public interface IReminder extends INotebookEntry
{
	/**
	*	The owning PlantSpecies.
	*
	*	@return	the PlantSpecies
	*/
	public Optional<IPlantSpecies> getPlantSpecies();

	/**
	*	The owning PlantVariety.
	*
	*	@return	the PlantVariety
	*/
	public Optional<IPlantVariety> getPlantVariety();

	/**
	*	The owning HusbandryClass.
	*
	*	@return	the HusbandryClass
	*/
	public Optional<IHusbandryClass> getHusbandryClass();

	/**
	*	The owning GroundworkActivity.
	*
	*	@return	the GroundworkActivity
	*/
	public Optional<IGroundworkActivity> getGroundworkActivity();

	/**
	 *	The owning Husbandry.
	 *	Used when a ToDoList is converted (back) into a Reminder
	 *
	 *	@return	the Husbandry
	 */
	public Optional<IHusbandry> getHusbandry();

	/**
	 *	The owning Purchase Item.
	 *	Used when a ToDoList is converted (back) into a Reminder
	 *
	 *	@return	the Husbandry
	 */
	public Optional<IPurchaseItem> getPurchaseItem();

	/**
	*	Get the value of showFrom.
	*
	*	@return	the value of showFrom
	*/
	public LocalDate getShowFrom();

	/**
	*	Get the value of singleShot.
	*
	*	@return	the value of singleShot
	*/
	public boolean isSingleShot();

	/**
	*	Get the value of repeatInterval (if present).
	*
	*	@return	the value of repeatInterval
	*/
	public Optional<String> getRepeatInterval();

	/**
	*	Get the value of repeatQuantifier (if present).
	*
	*	@return	the value of repeatQuantifier
	*/
	public Optional<Integer> getRepeatQuantifier();

	/**
	*	Get the value of repeatUntil (if present).
	*
	*	@return	the value of repeatUntil
	*/
	public Optional<LocalDate> getRepeatUntil();

	/**
	*	Get the value of description (if present).
	*
	*	@return	the value of description
	*/
	public Optional<String> getDescription();

	/**
	*	Get the value of lastUpdated.
	*
	*	@return	the value of lastUpdated
	*/
	public LocalDateTime getLastUpdated();

	/**
	*	Get the value of created.
	*
	*	@return	the value of created
	*/
	public LocalDateTime getCreated();


	/**
	*	Add a change listener to the Reminder.
	*
	*	@param	propertyName	identifies the change being flagged.  One of:<UL>
	*			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
	*			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
	*			<LI>Comment	-	this comment has been added to or deleted from this Reminder, newValue is an added comment, oldValue is a deleted comment
	</UL>
	*	@param	listener	a PropertyChangeListener (may be a Lambda expression)
	*/
	public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

	/**
	*	Remove a change listener from the Reminder.
	*
	*	@param	propertyName	identifies the change being flagged.  One of:<UL>
	*			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
	*			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
	*			<LI>Comment	-	this comment has been added to or deleted from this Reminder, newValue is an added comment, oldValue is a deleted comment
	</UL>
	*	@param	listener	the listener to remove
	*/
	public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

	/**
	*	Signal to all listeners that this Reminder has been deleted.
	*
	*/
	public void flagDeleted();

	/**
	*	Signal to all listeners that this Reminder has been replaced.
	*
	*	@param	newValue	the replacement Reminder
	*/
	public void flagReplaced(IReminder newValue);

}

