/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.spi;

import java.beans.PropertyChangeListener;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Year;
import java.util.Optional;

/**
 *	Free text Review article for the year.
  * Can cover everything or specific topics such as the performsnce of a new plant variety
 *  in a certain location or the effectiveness of a new pest control mechanism.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	3.0.0
 */
public interface IReview extends INotebookEntry
{

    /**
     *	Get the value of date.
     *
     *	@return	the value of date
     */
    public LocalDate getDate();

    /**
     *	Get the Year covered by this Review.
     *
     *	@return	the value of yearInReview
     */
    public Year getYearInReview();

    /**
     *	Get the actual period covered by the Review.
     *
     *	@return	the start of the period covered.
     */
    public Optional<LocalDate> getCoverFrom();
    /**
     *	Get the actual period covered by the Review.
     *
     *	@return	the end of the period covered.
     */
    public Optional<LocalDate> getCoverTo();

    /**
     * Get the title of the Journal entry.
     *
     * @return  the title of the Journal entry.
     */
    public String getTitle();

    /**
     * Get the text of the Journal entry.
     *
     * @return  the text of the Journal entry.
     */
    public Optional<String> getDescription();

    /**
     *	Get the value of lastUpdated.
     *
     *	@return	the value of lastUpdated
     */
    public LocalDateTime getLastUpdated();

    /**
     *	Get the value of created.
     *
     *	@return	the value of created
     */
    public LocalDateTime getCreated();

    /**
     *	Add a change listener to the Review entry.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this Husbandry, newValue is an added comment, oldValue is a deleted comment
     *			<LI>ToDoList	-	this ToDoList has been added to or deleted from this Husbandry's children, newValue is an added object, oldValue is the deleted object
    </UL>
     *	@param	listener	a PropertyChangeListener (may be a Lambda expression)
     */
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Remove a change listener from the Review entry.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this Husbandry, newValue is an added comment, oldValue is a deleted comment
     *			<LI>ToDoList	-	this ToDoList has been added to or deleted from this Husbandry's children, newValue is an added object, oldValue is the deleted object
    </UL>
     *	@param	listener	the listener to remove
     */
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Signal to all listeners that this Review entry has been deleted.
     *
     */
    public void flagDeleted();

    /**
     *	Signal to all listeners that this Review entry has been replaced.
     *
     *	@param	newValue	the replacement Review
     */
    public void flagReplaced(IReview newValue);

}
