/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.spi;

import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.util.Optional;

/**
 *	The various items referenced in a Review.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	3.0.0
 */
public interface IReviewReferences extends INotebookEntry
{
    /**
     *	The References are for this Review.
     *
     *	@return	the Review
     */
    public Optional<IReview> getReview();

    /**
     *	The Review concerns this PlantSpecies.
     *
     *	@return	the PlantSpecies
     */
    public Optional<IPlantSpecies> getPlantSpecies();

    /**
     *	The Review concerns this PlantVariety.
     *
     *	@return	the PlantVariety
     */
    public Optional<IPlantVariety> getPlantVariety();

    /**
     *	The Review concerns this HusbandryClass.
     *
     *	@return	the HusbandryClass
     */
    public Optional<IHusbandryClass> getHusbandryClass();

    /**
     *	The Review concerns this activity.
     *
     *	@return	the GroundworkActivity
     */
    public Optional<IGroundworkActivity> getGroundworkActivity();

    /**
     *	The Review concerns this  Affliction or its treatment.
     *
     *	@return	the Affliction
     */
    public Optional<IAffliction> getAffliction();

    /**
     *	The Review concerns this Location.
     *
     *	@return	the Location
     */
    public Optional<ILocation> getLocation();

    /**
     *	The Review concerns this Product.
     *
     *	@return	the Product
     */
    public Optional<IProduct> getProduct();

    /**
     *	Get the value of lastUpdated.
     *
     *	@return	the value of lastUpdated
     */
    public LocalDateTime getLastUpdated();

    /**
     *	Get the value of created.
     *
     *	@return	the value of created
     */
    public LocalDateTime getCreated();

    /**
     *	Add a change listener to the ReviewReferences entry.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this Husbandry, newValue is an added comment, oldValue is a deleted comment
     *			<LI>ToDoList	-	this ToDoList has been added to or deleted from this Husbandry's children, newValue is an added object, oldValue is the deleted object
    </UL>
     *	@param	listener	a PropertyChangeListener (may be a Lambda expression)
     */
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Remove a change listener from the ReviewReferences entry.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this Husbandry, newValue is an added comment, oldValue is a deleted comment
     *			<LI>ToDoList	-	this ToDoList has been added to or deleted from this Husbandry's children, newValue is an added object, oldValue is the deleted object
    </UL>
     *	@param	listener	the listener to remove
     */
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Signal to all listeners that this ReviewReferences entry has been deleted.
     *
     */
    public void flagDeleted();

    /**
     *	Signal to all listeners that this ReviewReferences entry has been replaced.
     *
     *	@param	newValue	the replacement ReviewReferences
     */
    public void flagReplaced(IReviewReferences newValue);

}
