/*
 * Copyright (C) 2018, 2019 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    2.5.0   First proper implementation
            Support selection by PlantSpecies and Variety
 */

package uk.co.gardennotebook.spi;

import java.util.List;

import java.time.*;

import java.math.BigDecimal;

/**
*   A sale of produce.
*
*	@author	Andy Gegg
*	@version	2.5.0
*	@since	1.0
*/
public interface ISaleLister
{
	/**
	*	Returns the Sale entries matching the given criteria.
	*	If no criteria have been specified all Sale entries will be returned.
	*	Criteria are applied additively (i.e. criterion AND criterion).
	*
	*	@return	A list of Sale entries matching previously specified criteria.
	*
	*	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	*			The causal Exception can be retrieved via <code>getCause()</code>
	*/
	public List<ISale> fetch() throws GNDBException;

	/**
	*	Select only these Sale entries
	*	May be called repeatedly to add to the list of Sale entries to be returned
	*
	*	@param items	the list of Sale entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ISaleLister sale(ISale... items);

	/**
	*	Select only these Sales with SaleItem entries
	*	May be called repeatedly to add to the list of Sale entries to be returned
	*
	*	@param items	the list of Sale entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ISaleLister sale(List<ISale> items);

	/**
	*	Select only those Sales with SaleItem entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Sales with SaleItem entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ISaleLister plantSpecies(IPlantSpecies... items);

	/**
	*	Select only those Sales with SaleItem entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Sales with SaleItem entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ISaleLister plantVariety(IPlantVariety... items);

	/**
	*	Select only those Sales with SaleItem entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Sales with SaleItem entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ISaleLister plantVariety(List<IPlantVariety> items);

	/**
	*	Select only those Sale entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Sales with SaleItem entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ISaleLister plantSpecies(List<IPlantSpecies> items);

	/**
	*	Select only those Sale entries which are on or after this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no earliest date will be used in selection
	*
	*	@param	date	the earliest date to match
	*	@return	this Lister
	*/
	public ISaleLister fromDate(LocalDate date);
	/**
	*	Select only those Sale entries which are on or before this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no latest date will be used in selection
	*
	*	@param	date	the latest date to match
	*	@return	this Lister
	*/
	public ISaleLister toDate(LocalDate date);

}
