/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Support Location field
 */

package uk.co.gardennotebook.spi;

import java.util.List;

import java.time.*;

/**
A noteworthy observation of wildlife - an interesting event, unusual species, etc
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	1.0
*/
public interface IWildlifeLister
{
	/**
	*	Returns the Wildlife entries matching the given criteria.
	*	If no criteria have been specified all Wildlife entries will be returned.
	*	Criteria are applied additively (i.e. criterion AND criterion).
	*
	*	@return	A list of Wildlife entries matching previously specified criteria.
	*
	*	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	*			The causal Exception can be retrieved via <code>getCause()</code>
	*/
	public List<IWildlife> fetch() throws GNDBException;

	/**
	*	Select only these Wildlife entries
	*	May be called repeatedly to add to the list of Wildlife entries to be returned
	*
	*	@param items	the list of Wildlife entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IWildlifeLister wildlife(IWildlife... items);

	/**
	*	Select only these Wildlife entries
	*	May be called repeatedly to add to the list of Wildlife entries to be returned
	*
	*	@param items	the list of Wildlife entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IWildlifeLister wildlife(List<IWildlife> items);

	/**
	*	Select only those Wildlife entries which are for these WildlifeSpecies entries.
	*	May be called repeatedly to add to the list of WildlifeSpecies entries to be selected.
	*
	*	@param items	the list of WildlifeSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IWildlifeLister wildlifeSpecies(IWildlifeSpecies... items);

	/**
	*	Select only those Wildlife entries which are for these WildlifeSpecies entries.
	*	May be called repeatedly to add to the list of WildlifeSpecies entries to be selected
	*
	*	@param items	the list of WildlifeSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IWildlifeLister wildlifeSpecies(List<IWildlifeSpecies> items);

	/**
	 *	Select only those Wildlife entries which are for these Locations.
	 *	May be called repeatedly to add to the list of Locations to be selected.
	 *
	 *	@param items	the list of Locations.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IWildlifeLister location(ILocation... items);

	/**
	 *	Select only those Wildlife entries which are for these Locations.
	 *	May be called repeatedly to add to the list of Locations to be selected
	 *
	 *	@param items	the list of Locations.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IWildlifeLister location(List<ILocation> items);

	/**
	*	Select only those Wildlife entries which are on or after this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no earliest date will be used in selection
	*
	*	@param	date	the earliest date to match
	*	@return	this Lister
	*/
	public IWildlifeLister fromDate(LocalDate date);
	/**
	*	Select only those Wildlife entries which are on or before this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no latest date will be used in selection
	*
	*	@param	date	the latest date to match
	*	@return	this Lister
	*/
	public IWildlifeLister toDate(LocalDate date);

}
