/*
 * Copyright (C) 2018, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeListenerProxy;
import java.beans.PropertyChangeSupport;

import uk.co.gardennotebook.spi.*;

import java.time.*;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonObjectBuilder;
import jakarta.json.JsonObject;

/**
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

final class Comment implements IComment
{
	private final PropertyChangeSupport pcs = new PropertyChangeSupport(this);

	private final int id;
	private final int ownerId;

	/*
	*	The comment is associated with a DB record of this type.
 * PC product category
 * BR product brand
 * PR product
 * SU supplier (retailer)
 * RP retailer has product
 * PI purchase item
 * PU purchase
 * SL shopping list
 * PS plant species
 * PV plant variety
 * PN plant note
 * PY plant synonym (no longer used)
 * HU husbandry
 * HC husbandry class
 * GA groundwork activity
 * GW groundwork
 * AC affliction class
 * AF affliction
 * AE affliction event
 * WC weather condition
 * WE weather
 * WD weather detail
 * WS wildlife species
 * WI wildlife
 * RM reminder
 * TD to-do list
 * H story line (thread)(no longer used)
 * CT comment
 * RS crop group sequence
 * CA cropping actual
 * CP cropping plan
 * JN journal
 * LN location
 * RW review
 * RR review references
 * SA sale
 * SI sale item
	*/
	private final String ownerType;
	private final LocalDate date;
	private final String comment;
	private final LocalDateTime lastUpdated;
	private final LocalDateTime created;
	private final NotebookEntryType ownerTypeEnum;

	/**
	*	Build an immutable Comment entry one field at a time
	*/
	Comment(
		final int id,
		final int ownerId,
		final String ownerType,
		final LocalDate date,
		final String comment,
		final LocalDateTime lastUpdated,
		final LocalDateTime created)
	{
		this.id = id;
		this.ownerId = ownerId;
		this.ownerType = ownerType;
		this.date = date;
		this.comment = comment;
		this.lastUpdated = lastUpdated;
		this.created = created;
		this.ownerTypeEnum = NotebookEntryType.fromString(ownerType);
	}

	/**
	*	Build an immutable Comment entry from a JSON dump.
	*	The dumped object must be complete (all non-nullable fields must have values) except
	*	the id field can be null or absent to indicate that this is a new item to be inserted.
	*
	*	@param	json	a JsonObject holding all the fields for a full initialisation.
	*/
	Comment(JsonObject json)
	{
		this.id = json.getInt("id", -1);
		this.ownerId = json.getInt("ownerId");
		this.ownerType = json.getString("ownerType");
		this.date = LocalDate.parse(json.getString("date"));
		this.comment = json.getString("comment");
		this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
		this.created = LocalDateTime.parse(json.getString("created"));
		this.ownerTypeEnum = NotebookEntryType.fromString(json.getString("ownerType"));
	}	//constructor from JSON

	int getId()
	{
		return id;
	}
	@Override
	public Integer getKey()
	{
		return id;
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.COMMENT;
	}

	@Override
	public Integer getOwnerKey()
	{
		return ownerId;
	}
	int getOwnerId()
	{
		return ownerId;
	}

	@Override
	public NotebookEntryType getOwnerType()
	{
		return ownerTypeEnum;
	}
	String getOwnerString()
	{
		return ownerType;
	}

	@Override
	public LocalDate getDate()
	{
		return date;
	}

	@Override
	public String getComment()
	{
		return comment;
	}

	@Override
	public LocalDateTime getLastUpdated()
	{
		return lastUpdated;
	}

	@Override
	public LocalDateTime getCreated()
	{
		return created;
	}

	JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
	{
		JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
		jsonBuilder.add("id", id);
		jsonBuilder.add("ownerId", ownerId);
		jsonBuilder.add("ownerType", ownerType);
		jsonBuilder.add("date", date.toString());
		jsonBuilder.add("comment", comment);
		jsonBuilder.add("lastUpdated", lastUpdated.toString());
		jsonBuilder.add("created", created.toString());
		jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "Comment");
		return jsonBuilder;
	}	//	toJson

	@Override
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		this.pcs.addPropertyChangeListener(propertyName, listener);
	}

	@Override
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		this.pcs.removePropertyChangeListener(propertyName, listener);
	}

	@Override
	public void flagDeleted()
	{
		this.pcs.firePropertyChange("deleted", this, null);
		PropertyChangeListener[] pcls = pcs.getPropertyChangeListeners();
		for (PropertyChangeListener pcl : pcls)
		{
			pcs.removePropertyChangeListener(pcl);
		}
	}

	@Override
	public void flagReplaced(final IComment newValue)
	{
		PropertyChangeListener[] pcls = pcs.getPropertyChangeListeners();
		for (PropertyChangeListener pcl : pcls)
		{
			if (pcl instanceof PropertyChangeListenerProxy proxy)
			{
				newValue.addPropertyChangeListener(proxy.getPropertyName(), proxy.getListener());
			}
		}
		this.pcs.firePropertyChange("replaced", this, newValue);
		for (PropertyChangeListener pcl : pcls)
		{
			pcs.removePropertyChangeListener(pcl);
		}
	}


	@Override
	public String toString() {
		return "Comment: " + "id: " + id + ", " +
				"ownerId: " + ownerId + ", " +
				"ownerType: " + ownerType + ", " +
				"date: " + date + ", " +
				"comment: " + comment + ", " +
				"lastUpdated: " + lastUpdated + ", " +
				"created: " + created;
	}

}
