/*
 * Copyright (C) 2018, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Location and Quantity added
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import jakarta.json.*;
import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.util.*;

/**
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	3.0.0
*/

final class CropRotationGroup implements ICropRotationGroup
{
	private final FlagHandler<ICropRotationGroup> flagHandler;
	{
		flagHandler = new FlagHandler<>(this);
	}

	private final int id;
	private final String name;
	private final String description;
	private final Integer subsequentCropGroupId;
	private final LocalDateTime lastUpdated;
	private final LocalDateTime created;
	private final List<Comment> commentsList;

	/**
	*	Build an immutable CropRotationGroup entry one field at a time
	*/
	CropRotationGroup(
		final int id,
		final String name,
		final String description,
		final Integer subsequentCropGroupId,
		final LocalDateTime lastUpdated,
		final LocalDateTime created,
		final Comment... comments)
	{
		this.id = id;
		this.name = name;
		this.subsequentCropGroupId = subsequentCropGroupId;
		this.description = description;
		this.lastUpdated = lastUpdated;
		this.created = created;

		if (comments != null && comments.length>0)
		{
			this.commentsList = new ArrayList<>(Arrays.asList(comments));
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}

	/**
	*	Build an immutable CropRotationGroup entry cloning the given CropRotationGroup entry but adding the comments list
	*/
	CropRotationGroup(
		final CropRotationGroup toCopy,
		final Comment... comments)
	{
		this(toCopy, Arrays.asList(comments));
	}

	/**
	*	Build an immutable CropRotationGroup entry cloning the given CropRotationGroup entry but adding the comments list
	*/
	CropRotationGroup(
		final CropRotationGroup toCopy,
		final List<Comment> comments)
	{
		this.id = toCopy.id;
		this.name = toCopy.name;
		this.subsequentCropGroupId = toCopy.subsequentCropGroupId;
		this.description = toCopy.description;
		this.lastUpdated = toCopy.lastUpdated;
		this.created = toCopy.created;
		if (comments != null && comments.size()>0)
		{
			if (toCopy.commentsList.size()>0)
			{
				// append new comments to previous list
				this.commentsList = new ArrayList<>(toCopy.commentsList);
				this.commentsList.addAll(comments);
			}
			else
			{	// no comments on original item
				this.commentsList = new ArrayList<>(comments);
			}
		}
		else
		{	// no new comments to add
			this.commentsList = toCopy.commentsList;
		}
	}

	/**
	*	Build an immutable CropRotationGroup entry from a JSON dump.
	*	The dumped object must be complete (all non-nullable fields must have values) except
	*	the id field can be null or absent to indicate that this is a new item to be inserted.
	*
	*	@param	json	a JsonObject holding all the fields for a full initialisation.
	*/
	CropRotationGroup(JsonObject json)
	{
		this.id = json.getInt("id", -1);
		this.name = json.getString("name");
		if (json.containsKey("description") && !json.isNull("description"))
		{
			this.description = json.getString("description");
		}
		else
		{
			this.description = null;
		}

//		this.subsequentCropGroupId = json.getInt("subsequentCropGroupId");
		if (json.containsKey("subsequentCropGroupId") && !json.isNull("subsequentCropGroupId"))
		{
			this.subsequentCropGroupId = json.getInt("subsequentCropGroupId");
		}
		else
		{
			this.subsequentCropGroupId = null;
		}

		this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
		this.created = LocalDateTime.parse(json.getString("created"));
		JsonArray jsonComments = json.getJsonArray("comments");
		if (jsonComments != null && !jsonComments.isEmpty())
		{// there is probably only one comment
			this.commentsList = new ArrayList<>(jsonComments.size());
			for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
			{
				this.commentsList.add(new Comment(ct));
			}
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}	//constructor from JSON

	int getId()
	{
		return id;
	}
	@Override
	public Integer getKey()
	{
		return id;
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.CROPROTATIONGROUP;
	}

	@Override
	public String getName()
	{
		return name;
	}

	@Override
	public Optional<String> getDescription()
	{
		return Optional.ofNullable(description);
	}

	int getSubsequentCropGroupId()
	{
		return subsequentCropGroupId;
	}
	@Override
	public Optional<ICropRotationGroup> getSubsequentCropGroup()
	{
		return Optional.ofNullable(MySQLCache.cacheCropRotationGroup.get(subsequentCropGroupId) );
	}

	@Override
	public LocalDateTime getLastUpdated()
	{
		return lastUpdated;
	}

	@Override
	public LocalDateTime getCreated()
	{
		return created;
	}

	@Override
	public boolean sameAs(INotebookEntry other)
	{
		if (other == null || other.getType() != this.getType())
		{
			return false;
		}
		return other.getKey().equals(this.getKey());
	}

	@Override
	public IPlantSpeciesLister getPlantSpecies()
	{
		return new PlantSpeciesLister().cropRotationGroup(this);
	}

	@Override
	public List<IComment> getComments() {
		return new ArrayList<>(this.commentsList);
	}

	JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
	{
		JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
		jsonBuilder.add("id", id);
		jsonBuilder.add("name", name);
		if (description != null)
		{
			jsonBuilder.add("description", description);
		}
		else
		{
			jsonBuilder.addNull("description");
		}

//		jsonBuilder.add("subsequentCropGroupId", subsequentCropGroupId);
		if (subsequentCropGroupId != null)
		{
			jsonBuilder.add("subsequentCropGroupId", subsequentCropGroupId);
		}
		else
		{
			jsonBuilder.addNull("subsequentCropGroupId");
		}
		jsonBuilder.add("lastUpdated", lastUpdated.toString());
		jsonBuilder.add("created", created.toString());
		if (commentsList != null && !commentsList.isEmpty())
		{// no point writing an empty comments array (the loaders handle this)
			JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
			for (Comment ct : commentsList)
			{
				jsonComments.add(ct.toJson(jsonFactory));
			}
			jsonBuilder.add("comments", jsonComments);
		}
		jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "CropRotationGroup");
		return jsonBuilder;
	}	//	toJson

	@Override
	public void addPropertyChangeListener( String propertyName,  PropertyChangeListener listener)
	{
		flagHandler.addPropertyChangeListener(propertyName, listener);
	}

	@Override
	public void removePropertyChangeListener( String propertyName,  PropertyChangeListener listener)
	{
		flagHandler.removePropertyChangeListener(propertyName, listener);
	}

	@Override
	public void flagDeleted()
	{
		flagHandler.flagDeleted();
	}

	@Override
	public void flagReplaced( ICropRotationGroup newValue)
	{
		flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
	}

	@Override
	public void flagChildDeleted( IPlantSpecies child)
	{
		flagHandler.flagChildDeleted("PlantSpecies", child);	}

	@Override
	public void flagChildAdded( IPlantSpecies child)
	{
		flagHandler.flagChildAdded("PlantSpecies", child);
	}

	@Override
	public void flagChildDeleted(ICroppingPlan child)
	{
		flagHandler.flagChildDeleted("PlantSpecies", child);	}

	@Override
	public void flagChildAdded(ICroppingPlan child)
	{
		flagHandler.flagChildAdded("PlantSpecies", child);
	}

	@Override
	public String toString() {
		return "CropRotationGroup: " + "id: " + id + ", " +
				"name: " + name + ", " +
				"description: " + description + ", " +
				"subsequentCropGroupId: " + subsequentCropGroupId + ", " +
				"lastUpdated: " + lastUpdated + ", " +
				"created: " + created;
	}

}
