/*
 * Copyright (C) 2018, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Use FLagHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import java.beans.PropertyChangeListener;
//import java.beans.PropertyChangeListenerProxy;
//import java.beans.PropertyChangeSupport;

import uk.co.gardennotebook.spi.*;

import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;

import java.time.*;
import java.util.Optional;

import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonArray;
import jakarta.json.JsonObjectBuilder;
import jakarta.json.JsonObject;

/**
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

final class HusbandryClass implements IHusbandryClass
{
//	private final PropertyChangeSupport pcs = new PropertyChangeSupport(this);
	private final FlagHandler<IHusbandryClass> flagHandler;
	{
		flagHandler = new FlagHandler<>(this);
	}

	private final int id;
	private final String name;
	private final String description;

	/*
	*	If true, this represents an end of life event.<BR>
For instance, death, discarded, abandoned, failed.
	*/
	private final boolean terminal;
	private final LocalDateTime lastUpdated;
	private final LocalDateTime created;
	private final List<Comment> commentsList;

	/**
	*	Build an immutable HusbandryClass entry one field at a time
	*/
	HusbandryClass(
		final int id,
		final String name,
		final String description,
		final boolean terminal,
		final LocalDateTime lastUpdated,
		final LocalDateTime created,
		final Comment... comments)
	{
		this.id = id;
		this.name = name;
		this.description = description;
		this.terminal = terminal;
		this.lastUpdated = lastUpdated;
		this.created = created;
		if (comments != null && comments.length>0)
		{
			this.commentsList = new ArrayList<>(Arrays.asList(comments));
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}

	/**
	*	Build an immutable HusbandryClass entry cloning the given HusbandryClass entry but adding the comments list
	*/
	HusbandryClass(
		final HusbandryClass toCopy,
		final Comment... comments)
	{
		this(toCopy, Arrays.asList(comments));
	}

	/**
	*	Build an immutable HusbandryClass entry cloning the given HusbandryClass entry but adding the comments list
	*/
	HusbandryClass(
		final HusbandryClass toCopy,
		final List<Comment> comments)
	{
		this.id = toCopy.id;
		this.name = toCopy.name;
		this.description = toCopy.description;
		this.terminal = toCopy.terminal;
		this.lastUpdated = toCopy.lastUpdated;
		this.created = toCopy.created;
		if (comments != null && comments.size()>0)
		{
			if (toCopy.commentsList.size()>0)
			{
				// append new comments to previous list
				this.commentsList = new ArrayList<>(toCopy.commentsList);
				this.commentsList.addAll(comments);
			}
			else
			{	// no comments on original item
				this.commentsList = new ArrayList<>(comments);
			}
		}
		else
		{	// no new comments to add
			this.commentsList = toCopy.commentsList;
		}
	}

	/**
	*	Build an immutable HusbandryClass entry from a JSON dump.
	*	The dumped object must be complete (all non-nullable fields must have values) except
	*	the id field can be null or absent to indicate that this is a new item to be inserted.
	*
	*	@param	json	a JsonObject holding all the fields for a full initialisation.
	*/
	HusbandryClass(JsonObject json)
	{
		this.id = json.getInt("id", -1);
		this.name = json.getString("name");
		if (json.containsKey("description") && !json.isNull("description"))
		{
			this.description = json.getString("description");
		}
		else
		{
			this.description = null;
		}

		this.terminal = json.getBoolean("terminal", false);
		this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
		this.created = LocalDateTime.parse(json.getString("created"));
		JsonArray jsonComments = json.getJsonArray("comments");
		if (jsonComments != null && !jsonComments.isEmpty())
		{// there is probably only one comment
			this.commentsList = new ArrayList<>(jsonComments.size());
			for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
			{
				this.commentsList.add(new Comment(ct));
			}
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}	//constructor from JSON

	int getId()
	{
		return id;
	}
	@Override
	public Integer getKey()
	{
		return id;
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.HUSBANDRYCLASS;
	}

	@Override
	public String getName()
	{
		return name;
	}

	@Override
	public Optional<String> getDescription()
	{
		return Optional.ofNullable(description);
	}

	@Override
	public boolean isTerminal()
	{
		return terminal;
	}

	@Override
	public LocalDateTime getLastUpdated()
	{
		return lastUpdated;
	}

	@Override
	public LocalDateTime getCreated()
	{
		return created;
	}

	@Override
	public boolean sameAs(INotebookEntry other)
	{
		if (other == null || other.getType() != this.getType())
		{
			return false;
		}
		return other.getKey().equals(this.getKey());
	}

	@Override
	public IHusbandryLister getHusbandry()
	{
		return new HusbandryLister().husbandryClass(this);
	}

	@Override
	public IReminderLister getReminder()
	{
		return new ReminderLister().husbandryClass(this);
	}

	@Override
	public IToDoListLister getToDoList()
	{
		return new ToDoListLister().husbandryClass(this);
	}

	@Override
	public List<IComment> getComments() {
		return new ArrayList<>(this.commentsList);
	}

	JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
	{
		JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
		jsonBuilder.add("id", id);
		jsonBuilder.add("name", name);
		if (description != null)
		{
			jsonBuilder.add("description", description);
		}
		else
		{
			jsonBuilder.addNull("description");
		}
		jsonBuilder.add("terminal", terminal);
		jsonBuilder.add("lastUpdated", lastUpdated.toString());
		jsonBuilder.add("created", created.toString());
		if (commentsList != null && !commentsList.isEmpty())
		{// no point writing an empty comments array (the loaders handle this)
			JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
			for (Comment ct : commentsList)
			{
				jsonComments.add(ct.toJson(jsonFactory));
			}
			jsonBuilder.add("comments", jsonComments);
		}
		jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "HusbandryClass");
		return jsonBuilder;
	}	//	toJson

	@Override
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.addPropertyChangeListener(propertyName, listener);
	}

	@Override
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.removePropertyChangeListener(propertyName, listener);
	}

	@Override
	public void flagDeleted()
	{
		flagHandler.flagDeleted();
	}

	@Override
	public void flagReplaced(final IHusbandryClass newValue)
	{
		flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
	}

	@Override
	public void flagChildDeleted(final IHusbandry child)
	{
		flagHandler.flagChildDeleted("Husbandry", child);
	}

	@Override
	public void flagChildAdded(final IHusbandry child)
	{
		flagHandler.flagChildAdded("Husbandry", child);
	}

	@Override
	public void flagChildDeleted(final IReminder child)
	{
		flagHandler.flagChildDeleted("Reminder", child);
	}

	@Override
	public void flagChildAdded(final IReminder child)
	{
		flagHandler.flagChildAdded("Reminder", child);
	}

	@Override
	public void flagChildDeleted(final IToDoList child)
	{
		flagHandler.flagChildDeleted("ToDoList", child);
	}

	@Override
	public void flagChildAdded(final IToDoList child)
	{
		flagHandler.flagChildAdded("ToDoList", child);
	}


	@Override
	public String toString() {
		return "HusbandryClass: " + "id: " + id + ", " +
				"name: " + name + ", " +
				"description: " + description + ", " +
				"terminal: " + terminal + ", " +
				"lastUpdated: " + lastUpdated + ", " +
				"created: " + created;
	}

}
