/*
 * Copyright (C) 2018, 2019, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.0.1   Reduced logging during processResults().
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.util.List;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import java.time.LocalDateTime;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonObjectBuilder;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

final class HusbandryClassLister implements IHusbandryClassLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private boolean useName = false;
	private String[] nameList = new String[10];
	private int nameNext = 0;	// next free slot in list

	private boolean useId = false;
	private int[] idList = new int[10];
	private int idNext = 0;	// next free slot in list

	HusbandryClassLister() {}

	@Override
	/**
	*{@inheritDoc}
	*/
	public List<IHusbandryClass> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		if (MySQLCache.invalidHusbandryClass)
		{
			load();
		}

		if(!useId && !useName)
		{
			useId = false;
			idNext = 0;
			useName = false;
			nameNext = 0;
			MySQLCache.invalidHusbandryClass = false;
			return MySQLCache.cacheHusbandryClass.values().stream().
					sorted((a, b) -> a.getName().compareToIgnoreCase(b.getName())).collect(Collectors.toList());
		}

		List<IHusbandryClass> outList = new ArrayList<>();

		idList = Arrays.copyOf(idList, idNext);

		if (useName)
		{
			nameList = Arrays.copyOf(nameList, nameNext);
			int [] nameKeys = MySQLCache.cacheHusbandryClass.values().stream().
				filter(c -> Arrays.stream(nameList).anyMatch(p -> p.equals(c.getName())))
				.mapToInt(INotebookEntry::getKey).
				toArray();
			idList = IntStream.concat(Arrays.stream(nameKeys), Arrays.stream(idList)).toArray();
		}

		idList = Arrays.stream(idList).distinct().toArray();

		for (int i : idList)
		{
			outList.add(MySQLCache.cacheHusbandryClass.get(i));

		}
		useId = false;
		idNext = 0;
		useName = false;
		nameNext = 0;
		MySQLCache.invalidHusbandryClass = false;

LOGGER.traceExit(log4jEntryMsg);
		return outList.stream().sorted((a,b) -> a.getName().compareToIgnoreCase(b.getName())).collect(Collectors.toList());
	}	// fetch()

	void load() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("load()");

		if (MySQLCache.invalidHusbandryClass)
		{
			List<HusbandryClass> tempList = new ArrayList<>();

            String query = "";
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = buildSQL_MySQL();
				case hsqldb -> query = buildSQL_hsqldb();
				case MSSQLServer -> query = buildSQL_MSSQLServer();
				default -> {
					LOGGER.debug("load(): no known rdbms");
					throw new GNDBException(new IllegalStateException("no known RDBMS"));
				}
			}
LOGGER.debug("load(): query: {}", query);
			try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
			{
				ResultSet rs = stmt.executeQuery(query);
				switch (DBConnection.DB_IN_USE)
				{
					case MariaDB, MySQL -> tempList = processResults_MySQL(rs);
					case hsqldb -> tempList = processResults_hsqldb(rs);
					case MSSQLServer -> tempList = processResults_MSSQLServer(rs);
					default -> {
						LOGGER.debug("fetch(): no known rdbms");
						throw new GNDBException(new IllegalStateException("no known RDBMS"));
					}
				}
				stmt.close();
			}catch (SQLException ex) {
				LOGGER.error("load(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
				throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
			}
			for (HusbandryClass ps : tempList)
			{
				MySQLCache.cacheHusbandryClass.putIfAbsent(ps.getId(), ps);
			}
		}

LOGGER.traceExit(log4jEntryMsg);
	}	// load()

    private String buildSQL_MySQL()
    {
		return "select d.*, c.* from husbandryclass as d " +
				"left join (select * from comment where ownerType = 'HC') as c " +
				"on d.husbandryClassId = c.ownerId " +
				" order by LOWER(d.name), c.date";
    }

    private String buildSQL_hsqldb()
    {
		return "select d.*, c.* from husbandryclass as d " +
				"left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'HC') as c " +
				"on d.husbandryClassId = c_ownerId " +
				" order by LOWER(d.name), c_date";
    }

    private String buildSQL_MSSQLServer()
    {
		return "select d.husbandryClassId as d_husbandryClassId, " +
				"d.name as d_name, " +
				"d.description as d_description, " +
				"d.terminal as d_terminal, " +
				"d.lastUpdated as d_lastUpdated, " +
				"d.created as d_created, " +
				" c.* from husbandryclass as d " + "left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'HC') as c " +
				"on d.husbandryClassId = c_ownerId " +
				" order by LOWER(d.name), c_date";
    }

	private List<HusbandryClass> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<HusbandryClass> tempList = new ArrayList<>();

		HusbandryClass item = null;

		while (rs.next())
		{
			int husbandryClassId = rs.getInt("d.husbandryClassId");
			String name = rs.getString("d.name");
			String description = rs.getString("d.description");
			boolean terminal = rs.getBoolean("d.terminal");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("processResults(): husbandryClassId: {},  name: {}, description: {}, terminal: {}, lastUpdated: {}, created: {}", husbandryClassId, name, description, terminal, lastUpdated, created); //  2.0.1
			if (item != null && husbandryClassId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c.commentId"),
					rs.getInt("c.ownerId"),
					"HC",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults(): extra comment is: {}", comm);
				item = new HusbandryClass(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new HusbandryClass(husbandryClassId, name, description, terminal, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						husbandryClassId,
						"HC",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults(): first comment is: {}", comm);
					item = new HusbandryClass(husbandryClassId, name, description, terminal, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<HusbandryClass> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<HusbandryClass> tempList = new ArrayList<>();

		HusbandryClass item = null;

		while (rs.next())
		{
			int husbandryClassId = rs.getInt("husbandryclass.husbandryClassId");
			String name = rs.getString("husbandryclass.name");
			String description = rs.getString("husbandryclass.description");
			boolean terminal = rs.getBoolean("husbandryclass.terminal");
			LocalDateTime lastUpdated = rs.getTimestamp("husbandryclass.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("husbandryclass.created").toLocalDateTime();
LOGGER.debug("processResults(): husbandryClassId: {},  name: {}, description: {}, terminal: {}, lastUpdated: {}, created: {}", husbandryClassId, name, description, terminal, lastUpdated, created); //  2.0.1
			if (item != null && husbandryClassId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"HC",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults(): extra comment is: {}", comm);
				item = new HusbandryClass(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new HusbandryClass(husbandryClassId, name, description, terminal, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						husbandryClassId,
						"HC",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults(): first comment is: {}", comm);
					item = new HusbandryClass(husbandryClassId, name, description, terminal, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<HusbandryClass> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<HusbandryClass> tempList = new ArrayList<>();

		HusbandryClass item = null;

		while (rs.next())
		{
			int husbandryClassId = rs.getInt("d_husbandryClassId");
			String name = rs.getString("d_name");
			String description = rs.getString("d_description");
			boolean terminal = rs.getBoolean("d_terminal");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("processResults(): husbandryClassId: {},  name: {}, description: {}, terminal: {}, lastUpdated: {}, created: {}", husbandryClassId, name, description, terminal, lastUpdated, created); //  2.0.1
			if (item != null && husbandryClassId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"HC",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults(): extra comment is: {}", comm);
				item = new HusbandryClass(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new HusbandryClass(husbandryClassId, name, description, terminal, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						husbandryClassId,
						"HC",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults(): first comment is: {}", comm);
					item = new HusbandryClass(husbandryClassId, name, description, terminal, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	void clear()
	{
		MySQLCache.cacheHusbandryClass.clear();
		MySQLCache.invalidHusbandryClass = true;
		MySQLCache.completeHusbandryClass = false;
	}

	/**
	*
	*Select only the HusbandryClass entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	HusbandryClassLister id(int... vals)
	{
		useId = true;
		if (idNext + vals.length >= idList.length)
		{
			idList = Arrays.copyOf(idList, idList.length+vals.length+10);
		}
		for (int val : vals)
		{
			idList[idNext++] = val;
		}
		return this;
	}

	@Override
	public IHusbandryClassLister name(String... vals)
	{
		if (vals == null) return this;
		if (vals.length == 0) return this;
		useName = true;
		if (nameNext + vals.length >= nameList.length)
		{
			nameList = Arrays.copyOf(nameList, nameList.length + vals.length + 10);
		}
		for (String val : vals) {nameList[nameNext++] = val;}
		return this;
	}

	@Override
	public IHusbandryClassLister key(IHusbandryClass... vals)
	{
		if (vals == null) return this;
		if (vals.length == 0) return this;
		int[] keys = new int[vals.length];
		int keyCount = 0;
		for (IHusbandryClass val : vals) {
			if (val == null) continue;
			Integer ky = val.getKey();
			if (ky == null) continue;
			keys[keyCount++] = ky;
		}
		if (keyCount == 0) return this;
		keys = Arrays.copyOf(keys, keyCount);	// trim array to actual size - should be a null-op
		return this.id(keys);
	}

	@Override
	public IHusbandryClassLister key(List<IHusbandryClass> items)
	{
		if (items == null) return this;
		if (items.isEmpty()) return this;
		return this.key(items.toArray(new IHusbandryClass[0]));
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		if (MySQLCache.invalidHusbandryClass)
		{
			load();
		}

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IHusbandryClass ihc : MySQLCache.cacheHusbandryClass.values())
		{
			HusbandryClass hc = (HusbandryClass)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "HusbandryClass");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "HusbandryClass.json"), false)))
		{
//			writer.writeArray(jsonHc.build());
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJSON(): IOException: ", ex);
		}
	}	// toJson

}
