/*
 *
 *  Copyright (C) 2021, 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
    Change log
	3.1.0	Use jakarta implementation of JSON
*/

package uk.co.gardennotebook.mysql;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.spi.*;

import jakarta.json.*;
import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.util.*;

/**
 *{@inheritDoc}
 *
 *	@author	Andy Gegg
 *	@version	3.1.0
 *	@since	3.0.0
 */

final class Location implements ILocation
{
//    private static final Logger LOGGER = LogManager.getLogger();

    private final FlagHandler<ILocation> flagHandler;
    {
        flagHandler = new FlagHandler<>(this);
    }

    private final int id;
    private final Integer parentLocationId;
    private final String name;
    private final String description;
    private final boolean underCover;
    private final String geometry;
    private final LocalDateTime lastUpdated;
    private final LocalDateTime created;
    private final List<Comment> commentsList;

    /**
     *  Build a Location entry one field at a time
     */
    Location(
            final int id,
            final Integer parentLocationId,
            final String name,
            final String description,
            final boolean underCover,
            final String geometry,
            final LocalDateTime lastUpdated,
            final LocalDateTime created,
            final Comment... comments)
    {
        this.id = id;
        this.parentLocationId = parentLocationId;
        this.name = name;
        this.description = description;
        this.underCover = underCover;
        this.geometry = geometry;
        this.lastUpdated = lastUpdated;
        this.created = created;
        if (comments != null && comments.length>0)
        {
            this.commentsList = new ArrayList<>(Arrays.asList(comments));
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }

    /**
     *	Build a Location entry cloning the given Location entry but adding the comments list
     */
    Location(
            final Location toCopy,
            final Comment... comments)
    {
        this(toCopy, Arrays.asList(comments));
    }

    /**
     *	Build a Location entry cloning the given Location entry but adding the comments list
     */
    Location(
            final Location toCopy,
            final List<Comment> comments)
    {
        this.id = toCopy.id;
        this.parentLocationId = toCopy.parentLocationId;
        this.name = toCopy.name;
        this.description = toCopy.description;
        this.underCover = toCopy.underCover;
        this.geometry = toCopy.geometry;
        this.lastUpdated = toCopy.lastUpdated;
        this.created = toCopy.created;
        if (comments != null && comments.size()>0)
        {
            if (toCopy.commentsList.size()>0)
            {
                // append new comments to previous list
                this.commentsList = new ArrayList<>(toCopy.commentsList);
                this.commentsList.addAll(comments);
            }
            else
            {	// no comments on original item
                this.commentsList = new ArrayList<>(comments);
            }
        }
        else
        {	// no new comments to add
            this.commentsList = toCopy.commentsList;
        }
    }

    /**
     *	Build a Location entry from a JSON dump.
     *	The dumped object must be complete (all non-nullable fields must have values) except
     *	the id field can be null or absent to indicate that this is a new item to be inserted.
     *
     *	@param	json	a JsonObject holding all the fields for a full initialisation.
     */
    Location(JsonObject json)
    {
//        EntryMessage log4jEntryMsg = LOGGER.traceEntry("JSON constructor");
        this.id = json.getInt("id", -1);
//        LOGGER.debug("id: {}", this.id);
        if (json.containsKey("parentLocationId") && !json.isNull("parentLocationId"))
        {
            this.parentLocationId = json.getInt("parentLocationId");
        }
        else
        {
            this.parentLocationId = null;
        }
//        LOGGER.debug("parentLocationId: {}", this.parentLocationId);

        this.name = json.getString("name");
//        LOGGER.debug("name: {}", this.name);
        if (json.containsKey("description") && !json.isNull("description"))
        {
            this.description = json.getString("description");
        }
        else
        {
            this.description = null;
        }
//        LOGGER.debug("description: {}", this.description);

        this.underCover = json.getBoolean("underCover");
//        LOGGER.debug("underCover: {}", this.underCover);

        if (json.containsKey("geometry") && !json.isNull("geometry"))
        {
//            LOGGER.debug("has found geometry field");
            this.geometry = json.getString("geometry");
        }
        else
        {
//            LOGGER.debug("no geometry field");
            this.geometry = null;
        }
//        LOGGER.debug("geometry: {}", this.geometry);

        this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
        this.created = LocalDateTime.parse(json.getString("created"));
//        LOGGER.debug("before comments");
        JsonArray jsonComments = json.getJsonArray("comments");
        if (jsonComments != null && !jsonComments.isEmpty())
        {// there is probably only one comment
            this.commentsList = new ArrayList<>(jsonComments.size());
            for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
            {
                this.commentsList.add(new Comment(ct));
            }
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
//        LOGGER.debug("commentsList: {}", this.commentsList);
    }	//constructor from JSON

    int getId()
    {
        return id;
    }

    @Override
    public Integer getKey()
    {
        return id;
    }

    @Override
    public NotebookEntryType getType()
    {
        return NotebookEntryType.LOCATION;
    }

    Integer getParentLocationId()
    {
        return parentLocationId;
    }
    @Override
    public Optional<ILocation> getParentLocation()
    {
        return Optional.ofNullable(MySQLCache.cacheLocation.get(parentLocationId));
    }

    @Override
    public boolean hasParent()
    {
        return parentLocationId != null;
    }

    @Override
    public String getName()
    {
        return name;
    }

    @Override
    public Optional<String> getDescription()
    {
        return Optional.ofNullable(description);
    }

    @Override
    public boolean isUnderCover()
    {
        return underCover;
    }

    @Override
    public Optional<String> getGeometry()
    {
        return Optional.ofNullable(geometry);
    }

    @Override
    public LocalDateTime getLastUpdated()
    {
        return lastUpdated;
    }

    @Override
    public LocalDateTime getCreated()
    {
        return created;
    }

    @Override
    public boolean sameAs(INotebookEntry other)
    {
        if (other == null || other.getType() != this.getType())
        {
            return false;
        }
        return other.getKey().equals(this.getKey());
    }

    @Override
    public IAfflictionEventLister getAfflictionEvent()
    {
        return new AfflictionEventLister().location(this);
    }

    @Override
    public IGroundworkLister getGroundwork()
    {
        return new GroundworkLister().location(this);
    }

    @Override
    public IHusbandryLister getHusbandry()
    {
        return new HusbandryLister().location(this);
    }

    @Override
    public List<IComment> getComments()
    {
        return new ArrayList<>(this.commentsList);
    }

    JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
    {
        JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
        jsonBuilder.add("id", id);
        if (parentLocationId != null)
        {
            jsonBuilder.add("parentLocationId", parentLocationId);
        }
        else
        {
            jsonBuilder.addNull("parentLocationId");
        }
        jsonBuilder.add("name", name);
        if (description != null)
        {
            jsonBuilder.add("description", description);
        }
        else
        {
            jsonBuilder.addNull("description");
        }
        jsonBuilder.add("underCover", underCover);
        if (geometry != null)
        {
            jsonBuilder.add("geometry", geometry);
        }
        else
        {
            jsonBuilder.addNull("geometry");
        }
        jsonBuilder.add("lastUpdated", lastUpdated.toString());
        jsonBuilder.add("created", created.toString());
        if (commentsList != null && !commentsList.isEmpty())
        {// no point writing an empty comments array (the loaders handle this)
            JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
            for (Comment ct : commentsList)
            {
                jsonComments.add(ct.toJson(jsonFactory));
            }
            jsonBuilder.add("comments", jsonComments);
        }
        jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "Location");
        return jsonBuilder;
    }	//	toJson

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.removePropertyChangeListener(propertyName, listener);
    }

    @Override
    public void flagDeleted()
    {
        flagHandler.flagDeleted();
    }

    @Override
    public void flagReplaced(final ILocation newValue)
    {
        flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
    }

    @Override
    public void flagChildDeleted(ILocation child)
    {
        flagHandler.flagChildDeleted("Location", child);
    }

    @Override
    public void flagChildAdded(ILocation child)
    {
        flagHandler.flagChildAdded("Location", child);
    }

    @Override
    public void flagChildDeleted(IAfflictionEvent child)
    {
        flagHandler.flagChildDeleted("AfflictionEvent", child);
    }

    @Override
    public void flagChildAdded(IAfflictionEvent child)
    {
        flagHandler.flagChildAdded("AfflictionEvent", child);
    }

    @Override
    public void flagChildDeleted(IGroundwork child)
    {
        flagHandler.flagChildDeleted("Groundwork", child);
    }

    @Override
    public void flagChildAdded(IGroundwork child)
    {
        flagHandler.flagChildAdded("Groundwork", child);
    }

    @Override
    public void flagChildDeleted(IHusbandry child)
    {
        flagHandler.flagChildDeleted("Husbandry", child);
    }

    @Override
    public void flagChildAdded(IHusbandry child)
    {
        flagHandler.flagChildAdded("Husbandry", child);
    }

    @Override
    public void flagChildDeleted(IReview child)
    {
        flagHandler.flagChildDeleted("Review", child);
    }

    @Override
    public void flagChildAdded(IReview child)
    {
        flagHandler.flagChildAdded("Review", child);
    }

    @Override
    public void flagChildDeleted(ICroppingPlan child)
    {
        flagHandler.flagChildDeleted("CroppingPlan", child);
    }

    @Override
    public void flagChildAdded(ICroppingPlan child)
    {
        flagHandler.flagChildAdded("CroppingPlan", child);
    }

    @Override
    public void flagChildDeleted(IWildlife child)
    {
        flagHandler.flagChildDeleted("Wildlife", child);
    }

    @Override
    public void flagChildAdded(IWildlife child)
    {
        flagHandler.flagChildAdded("WIldlife", child);
    }

    @Override
    public String toString()
    {
        return "Location: id: " + id + ", " +
            "parentLocationId: " + parentLocationId + ", " +
            "name: " + name +", " +
            "description: " + description + ", " +
            "underCover: " + underCover + ", " +
            "geometry: " + geometry + ", " +
            "lastUpdated: " + lastUpdated + ", " +
            "created: " + created;
    }
}
