/*
 * Copyright (C) 2018, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Use FlagHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import java.beans.PropertyChangeListener;

import uk.co.gardennotebook.spi.*;

import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;

import java.time.*;
import java.util.Optional;

import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonArray;
import jakarta.json.JsonObjectBuilder;
import jakarta.json.JsonObject;

/**
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

final class PlantNote implements IPlantNote
{
//	private final PropertyChangeSupport pcs = new PropertyChangeSupport(this);
	private final FlagHandler<IPlantNote> flagHandler;
	{
		flagHandler = new FlagHandler<>(this);
	}

	private final int id;

	/*
	*	This note is for a plant species.  Individual varieties may have their own notes as well as or instead of the variety's notes.
	*/
	private final int plantSpeciesId;

	/*
	*	If given, plantSpeciesId must also be given; this allows easy retrieval of notes for a species and any variety specific notes.
	*/
	private final Integer plantVarietyId;
	private final String title;
	private final String note;
	private final LocalDateTime lastUpdated;
	private final LocalDateTime created;
	private final List<Comment> commentsList;

	/**
	*	Build an immutable PlantNote entry one field at a time
	*/
	PlantNote(
		final int id,
		final int plantSpeciesId,
		final Integer plantVarietyId,
		final String title,
		final String note,
		final LocalDateTime lastUpdated,
		final LocalDateTime created,
		final Comment... comments)
	{
		this.id = id;
		this.plantSpeciesId = plantSpeciesId;
		this.plantVarietyId = plantVarietyId;
		this.title = title;
		this.note = note;
		this.lastUpdated = lastUpdated;
		this.created = created;
		if (comments != null && comments.length>0)
		{
			this.commentsList = new ArrayList<>(Arrays.asList(comments));
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}

	/**
	*	Build an immutable PlantNote entry cloning the given PlantNote entry but adding the comments list
	*/
	PlantNote(
		final PlantNote toCopy,
		final Comment... comments)
	{
		this(toCopy, Arrays.asList(comments));
	}

	/**
	*	Build an immutable PlantNote entry cloning the given PlantNote entry but adding the comments list
	*/
	PlantNote(
		final PlantNote toCopy,
		final List<Comment> comments)
	{
		this.id = toCopy.id;
		this.plantSpeciesId = toCopy.plantSpeciesId;
		this.plantVarietyId = toCopy.plantVarietyId;
		this.title = toCopy.title;
		this.note = toCopy.note;
		this.lastUpdated = toCopy.lastUpdated;
		this.created = toCopy.created;
		if (comments != null && comments.size()>0)
		{
			if (toCopy.commentsList.size()>0)
			{
				// append new comments to previous list
				this.commentsList = new ArrayList<>(toCopy.commentsList);
				this.commentsList.addAll(comments);
			}
			else
			{	// no comments on original item
				this.commentsList = new ArrayList<>(comments);
			}
		}
		else
		{	// no new comments to add
			this.commentsList = toCopy.commentsList;
		}
	}

	/**
	*	Build an immutable PlantNote entry from a JSON dump.
	*	The dumped object must be complete (all non-nullable fields must have values) except
	*	the id field can be null or absent to indicate that this is a new item to be inserted.
	*
	*	@param	json	a JsonObject holding all the fields for a full initialisation.
	*/
	PlantNote(JsonObject json)
	{
		this.id = json.getInt("id", -1);
		this.plantSpeciesId = json.getInt("plantSpeciesId");
		if (json.containsKey("plantVarietyId") && !json.isNull("plantVarietyId"))
		{
			this.plantVarietyId = json.getInt("plantVarietyId");
		}
		else
		{
			this.plantVarietyId = null;
		}

		this.title = json.getString("title");
		if (json.containsKey("note") && !json.isNull("note"))
		{
			this.note = json.getString("note");
		}
		else
		{
			this.note = null;
		}

		this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
		this.created = LocalDateTime.parse(json.getString("created"));
		JsonArray jsonComments = json.getJsonArray("comments");
		if (jsonComments != null && !jsonComments.isEmpty())
		{// there is probably only one comment
			this.commentsList = new ArrayList<>(jsonComments.size());
			for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
			{
				this.commentsList.add(new Comment(ct));
			}
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}	//constructor from JSON

	int getId()
	{
		return id;
	}
	@Override
	public Integer getKey()
	{
		return id;
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.PLANTNOTE;
	}

	int getPlantSpeciesId()
	{
		return plantSpeciesId;
	}
	@Override
	public IPlantSpecies getPlantSpecies()
	{
		return MySQLCache.cachePlantSpecies.get(plantSpeciesId);
	}

	Integer getPlantVarietyId()
	{
		return plantVarietyId;
	}
	@Override
	public Optional<IPlantVariety> getPlantVariety()
	{
		return Optional.ofNullable(MySQLCache.cachePlantVariety.get(plantVarietyId));
	}

	@Override
	public String getTitle()
	{
		return title;
	}

	@Override
	public Optional<String> getNote()
	{
		return Optional.ofNullable(note);
	}

	@Override
	public LocalDateTime getLastUpdated()
	{
		return lastUpdated;
	}

	@Override
	public LocalDateTime getCreated()
	{
		return created;
	}

	@Override
	public boolean sameAs(INotebookEntry other)
	{
		if (other == null || other.getType() != this.getType())
		{
			return false;
		}
		return other.getKey().equals(this.getKey());
	}

	@Override
	public List<IComment> getComments() {
		return new ArrayList<>(this.commentsList);
	}

	JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
	{
		JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
		jsonBuilder.add("id", id);
		jsonBuilder.add("plantSpeciesId", plantSpeciesId);
		if (plantVarietyId != null)
		{
			jsonBuilder.add("plantVarietyId", plantVarietyId);
		}
		else
		{
			jsonBuilder.addNull("plantVarietyId");
		}
		jsonBuilder.add("title", title);
		if (note != null)
		{
			jsonBuilder.add("note", note);
		}
		else
		{
			jsonBuilder.addNull("note");
		}
		jsonBuilder.add("lastUpdated", lastUpdated.toString());
		jsonBuilder.add("created", created.toString());
		if (commentsList != null && !commentsList.isEmpty())
		{// no point writing an empty comments array (the loaders handle this)
			JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
			for (Comment ct : commentsList)
			{
				jsonComments.add(ct.toJson(jsonFactory));
			}
			jsonBuilder.add("comments", jsonComments);
		}
		jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "PlantNote");
		return jsonBuilder;
	}	//	toJson

	@Override
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.addPropertyChangeListener(propertyName, listener);
	}

	@Override
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.removePropertyChangeListener(propertyName, listener);
	}

	@Override
	public void flagDeleted()
	{
		flagHandler.flagDeleted();
	}

	@Override
	public void flagReplaced(final IPlantNote newValue)
	{
		flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
	}


	@Override
	public String toString() {
		return "PlantNote: " + "id: " + id + ", " +
				"plantSpeciesId: " + plantSpeciesId + ", " +
				"plantVarietyId: " + plantVarietyId + ", " +
				"title: " + title + ", " +
				"note: " + note + ", " +
				"lastUpdated: " + lastUpdated + ", " +
				"created: " + created;
	}

}
