/*
 * Copyright (C) 2018, 2019, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.1.1   isCommonNameDuplicate() - use PreparedStatement in SELECT to guard String values
	2.2.0   Support hsqldb dialect
    2.2.5   Improve JSON load - make it faster!
    2.3.0   Retrieve generated keys properly!
    2.4.0   Support MS SQLServer
    3.0.0	Use DBCommentHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import uk.co.gardennotebook.util.StoryLineTree;

import java.sql.*;
//import java.sql.Types.*;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

import jakarta.json.JsonObject;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import org.apache.logging.log4j.Level;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class PlantSpeciesBuilder implements IPlantSpeciesBuilder
{
	private static final Logger LOGGER = LogManager.getLogger();

	private IPlantSpecies oldInstance = null;

	private final boolean newInstance;

	private boolean changedAncestor = false;

	private int id;

	/*
	*	The crop rotation group for this plant species
	 */
	private Integer cropRotationGroupId;
	private boolean changedCropRotationGroupId;

	/*
	*	The name by which plants of this type are usually known.
	*/
	private String commonName;
	private boolean changedCommonName = false;

	/*
	*	The formal horticultural name for this species.
	*	The common name is language specific, the latin name is internationally standardised.
	*/
	private String latinName;
	private boolean changedLatinName = false;
	private String description;
	private boolean changedDescription = false;

	/*
	*	The plant's function in the garden, typically vegetable, ornamental, weed.
	*/
	private String utility;
	private boolean changedUtility = false;

	/*
	*	The plant's hardiness or tenderness, typically hardy, half hardy, tender.
	*/
	private String hardiness;
	private boolean changedHardiness = false;

	/*
	*	Typically annual, biennial or perennial.  Variations such as 'perennial grown as annual' (e.g. runner bean).
	*/
	private String lifeType;
	private boolean changedLifeType = false;

	/*
	*	Such as climber, shrub, tree.
	*/
	private String plantType;
	private boolean changedPlantType = false;
	private LocalDateTime lastUpdated;
	private LocalDateTime created;
	private boolean somethingChanged = false;

private boolean changedComments = false;
	private DBCommentHandler commentHandler;	//	compiler will not allow declaration as final

	/**
	*	constructor to use for a new entry
	*/
	PlantSpeciesBuilder()
	{
		this(null);
	}

	/**
	*	constructor to use to edit or delete an existing entry
	*
	*	@param	oldVal	the existing item to modify or delete; if null a new entry will be created
	*/
	PlantSpeciesBuilder(final IPlantSpecies oldVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("constructor(): oldVal={}", oldVal);
		if (oldVal == null || oldVal.getKey() == null || !(oldVal.getKey() instanceof Integer))
		{
			newInstance = true;
			oldInstance = null;
			this.id = -1;
			commentHandler = new DBCommentHandler(NotebookEntryType.PLANTSPECIES);
			return;
		}

		newInstance = false;
		oldInstance = oldVal;

		PlantSpecies baseObj;
		if (oldVal instanceof PlantSpecies)
		{
			baseObj = (PlantSpecies)oldVal;
			this.id = baseObj.getId();
			this.cropRotationGroupId = baseObj.cropRotationGroupId;
			this.commonName = baseObj.getCommonName();
			this.latinName = baseObj.getLatinName().orElse(null);
			this.description = baseObj.getDescription().orElse(null);
			this.utility = baseObj.getUtility().orElse(null);
			this.hardiness = baseObj.getHardiness().orElse(null);
			this.lifeType = baseObj.getLifeType().orElse(null);
			this.plantType = baseObj.getPlantType().orElse(null);
			this.lastUpdated = baseObj.getLastUpdated();
			this.created = baseObj.getCreated();
		}
		else
		{
			Integer ky = oldVal.getKey();
			if (ky == null) return;
			this.id = ky;
			ICropRotationGroup crg = oldVal.getCropRotationGroup().orElse(null);
			if (crg == null)
			{
				this.cropRotationGroupId = 0;
			}
			else
			{
				this.cropRotationGroupId = crg.getKey();
			}
			this.commonName = oldVal.getCommonName();
			this.latinName = oldVal.getLatinName().orElse(null);
			this.description = oldVal.getDescription().orElse(null);
			this.utility = oldVal.getUtility().orElse(null);
			this.hardiness = oldVal.getHardiness().orElse(null);
			this.lifeType = oldVal.getLifeType().orElse(null);
			this.plantType = oldVal.getPlantType().orElse(null);
			this.lastUpdated = oldVal.getLastUpdated();
			this.created = oldVal.getCreated();
		}
		commentHandler = new DBCommentHandler(NotebookEntryType.PLANTSPECIES, this.id);
		LOGGER.traceExit();
	}	//	constructor()

	/**
	 * Give the new value of cropRotationGroupId.
	 *
	 * @param	newVal	the new value
	 */
	IPlantSpeciesBuilder cropRotationGroupId(final int newVal)
	{
		if (this.cropRotationGroupId == newVal) return this;
		this.cropRotationGroupId = newVal;
		changedCropRotationGroupId = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}
	@Override
	public IPlantSpeciesBuilder cropRotationGroup(final ICropRotationGroup newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("cropRotationGroup(): oldVal={}, newVal={}", this.cropRotationGroupId, newVal);
		if (newVal == null) return this;
		if (this.cropRotationGroupId == newVal.getKey()) return this;
		this.cropRotationGroupId = newVal.getKey();
		changedCropRotationGroupId = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public boolean isCommonNameDuplicate(final String newVal) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("isCommonNameDuplicate({})", newVal);

        if (newVal == null || newVal.isBlank())
            return false;
        
        if (!newInstance && !changedCommonName)
            return false;

        if (!newInstance && newVal.equals(oldInstance.getCommonName()))
            return false;

		boolean duplicateFound = false;
		String query;
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = "select exists (select 1 from plantspecies where commonName = ? ) as numvals";
			case hsqldb -> query = "select exists (select 1 from plantspecies where commonName = ? ) as numvals from (values(99))";
			case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from plantspecies where commonName = ? ) THEN 1 ELSE 0 END as numvals";
			default -> {
				LOGGER.error("isCommonNameDuplicate(): no known rdbms");
				throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: isCommonNameDuplicate(): no known RDBMS"));
			}
		}
LOGGER.debug("isCommonNameDuplicate(): query: {}", query);
		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query);
            )
		{
LOGGER.debug("isCommonNameDuplicate(): setParam 1: {}", newVal);
            stmt.setString(1, newVal);
			ResultSet rs = stmt.executeQuery();
            rs.next();
			duplicateFound = rs.getBoolean("numvals");
LOGGER.debug("isCommonNameDuplicate(): result: {}", duplicateFound);
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("isCommonNameDuplicate(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
		return LOGGER.traceExit(log4jEntryMsg, duplicateFound);
	}	// isCommonNameDuplicate()

	@Override
	public IPlantSpeciesBuilder commonName(final String newVal) throws IllegalArgumentException, GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("commonName(): oldVal={}, newVal={}", this.commonName, newVal);

		if (newVal == null) return this;
		if (newVal.equals(this.commonName)) return this;
//		if (isCommonNameDuplicate(newVal))
//		{
//			throw LOGGER.throwing(Level.DEBUG, new IllegalArgumentException("PlantSpeciesBuilder: commonName: " +newVal+" already exists"));
//		}
		this.commonName = newVal;
		changedCommonName = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPlantSpeciesBuilder latinName(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("latinName(): oldVal={}, newVal={}", this.latinName, newVal);

		if (newVal == null && this.latinName == null) return this;
		if (newVal != null && this.latinName != null && newVal.equals(this.latinName)) return this;
		this.latinName = newVal;
		changedLatinName = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPlantSpeciesBuilder description(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("description(): oldVal={}, newVal={}", this.description, newVal);

		if (newVal == null && this.description == null) return this;
		if (newVal != null && this.description != null && newVal.equals(this.description)) return this;
		this.description = newVal;
		changedDescription = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPlantSpeciesBuilder utility(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("utility(): oldVal={}, newVal={}", this.utility, newVal);

		if (newVal == null && this.utility == null) return this;
		if (newVal != null && this.utility != null && newVal.equals(this.utility)) return this;
		this.utility = newVal;
		changedUtility = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPlantSpeciesBuilder hardiness(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("hardiness(): oldVal={}, newVal={}", this.hardiness, newVal);

		if (newVal == null && this.hardiness == null) return this;
		if (newVal != null && this.hardiness != null && newVal.equals(this.hardiness)) return this;
		this.hardiness = newVal;
		changedHardiness = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPlantSpeciesBuilder lifeType(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("lifeType(): oldVal={}, newVal={}", this.lifeType, newVal);

		if (newVal == null && this.lifeType == null) return this;
		if (newVal != null && this.lifeType != null && newVal.equals(this.lifeType)) return this;
		this.lifeType = newVal;
		changedLifeType = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPlantSpeciesBuilder plantType(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("plantType(): oldVal={}, newVal={}", this.plantType, newVal);

		if (newVal == null && this.plantType == null) return this;
		if (newVal != null && this.plantType != null && newVal.equals(this.plantType)) return this;
		this.plantType = newVal;
		changedPlantType = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPlantSpeciesBuilder addComment(final String... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment[array]()");

		commentHandler.addComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit("addComment");
		return this;
	}

	@Override
	public IPlantSpeciesBuilder addComment(final List<String> newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment<list>()");

		commentHandler.addComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit("addComment");
		return this;
	}

	/**
	*	remove a comment from this item
	*
	*	@param	newVals	the comment to remove.  If the comment does not exist, this is a null-op
	*	@return	 this Builder
	*/
	IPlantSpeciesBuilder deleteComment(int... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPlantSpeciesBuilder deleteComment(final IComment... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPlantSpeciesBuilder deleteComment(final List<IComment> vals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(vals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPlantSpeciesBuilder changeComment(final IComment base, final String comment)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, comment);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPlantSpeciesBuilder changeComment(final IComment base, final LocalDate date)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, date);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPlantSpeciesBuilder changeComment(final IComment base, final LocalDate date, final String comment)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, date, comment);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPlantSpecies save() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("save(): somethingChanged: {}, newInstance: {}, changedComments: {}, changedAncestor: {}",
                                                                somethingChanged, newInstance, changedComments, changedAncestor);

		if (!somethingChanged && !newInstance && !changedComments && !changedAncestor)
		{
			LOGGER.traceExit("nothing changed");
			return MySQLCache.cachePlantSpecies.get(this.id);
		}

		if (newInstance)
		{
			doInsert();
		}
		else if (somethingChanged)
		{
			doUpdate();
		}

		if (changedComments)
		{
			commentHandler.setParentId(this.id);
			commentHandler.save();
		}

// mark cache as dirty
		MySQLCache.invalidPlantSpecies = true;
		if (!newInstance &&(somethingChanged || changedComments))
		{
			MySQLCache.cachePlantSpecies.remove(this.id);
		}
// populate the cache
		new PlantSpeciesLister().id(this.id).fetch();
		IPlantSpecies newValue = MySQLCache.cachePlantSpecies.get(this.id);
		if (oldInstance != null)
		{
			oldInstance.flagReplaced(newValue);
		}

		//	stop multiple saves!
		oldInstance = null;

		somethingChanged = false;
		changedComments = false;
		changedAncestor = false;
		changedCommonName = false;
		changedLatinName = false;
		changedDescription = false;
		changedUtility = false;
		changedHardiness = false;
		changedLifeType = false;
		changedPlantType = false;

		LOGGER.traceExit(log4jEntryMsg);
		return newValue;
	}	//	save()

	@Override
	public boolean needSave()
	{
		return somethingChanged || changedComments || changedAncestor;
	}	// needSave()

	@Override
	public boolean canSave()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("canSave(): newInstance={}", newInstance);

		if (!needSave())
		{//	save() will be a null-op but that's OK
			return true;
		}
		if (this.commonName == null ||
			this.commonName.isEmpty() )
		{
			LOGGER.debug("commonName not set");
			return false;
		}
        try
        {
            if (isCommonNameDuplicate(this.commonName)) 
            {
                LOGGER.debug("PlantSpeciesBuilder: canSave(): {} already exists", this.commonName);
                return false;
            }
        } catch (GNDBException ex) {
			throw LOGGER.throwing(Level.DEBUG, new IllegalArgumentException("PlantSpeciesBuilder: commonName: " +this.commonName+" already exists"));
        }
        
		return true;
	}	// canSave()

	@Override
	public boolean canDelete() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("canDelete(): newInstance={}", newInstance);

		if (newInstance) return LOGGER.traceExit(log4jEntryMsg, false);

		boolean  readValue = false;
		try (	Connection conn = DBConnection.getConnection();
				Statement stmt = conn.createStatement();	)
		{
            String query;
            ResultSet rs;

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from plantvariety where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from plantvariety where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from plantvariety where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from husbandry where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from husbandry where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from husbandry where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from afflictionevent where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from afflictionevent where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from afflictionevent where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from groundwork where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from groundwork where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from groundwork where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from plantnote where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from plantnote where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from plantnote where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from product where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from product where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from product where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from reminder where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from reminder where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from reminder where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from saleitem where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from saleitem where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from saleitem where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from todolist where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from todolist where plantSpeciesId = " + this.id +
						") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from todolist where plantSpeciesId = " + this.id +
						") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from reviewreferences where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from reviewreferences where plantSpeciesId = " + this.id + ") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from reviewreferences where plantSpeciesId = " + this.id + ") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
			LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
			rs.next();
			readValue = readValue || rs.getBoolean("fred");
			rs.close();
			LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}

			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from croppingactual where plantSpeciesId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from croppingactual where plantSpeciesId = " + this.id + ") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from croppingactual where plantSpeciesId = " + this.id + ") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PlantSpeciesBuilder: canDelete(): no known RDBMS"));
				}
			}
			LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
			rs.next();
			readValue = readValue || rs.getBoolean("fred");
			rs.close();
			LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}
			stmt.close();
            
		}catch (SQLException ex) {
			LOGGER.error("canDelete(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
		return LOGGER.traceExit(log4jEntryMsg, !readValue);
	}	// canDelete()

	@Override
	public void delete() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("delete(): newInstance={}", newInstance);

		if (newInstance) return;
		if (!canDelete()) return;

		int res = 0;
		String query = "delete from plantspecies where plantSpeciesId = " + this.id;
LOGGER.debug("delete(): query: {}", query);
		try (	Connection conn = DBConnection.getConnection();
				Statement stmt = conn.createStatement();	)
		{
			res = stmt.executeUpdate(query);
LOGGER.debug("delete(): result: {}", res);
			// tidy up dependencies with nullable keys
			if (res > 0) {
				query = "delete from comment where ownerId = " + this.id + " and ownerType = 'PS'";
				int res2 = stmt.executeUpdate(query);
LOGGER.debug("delete() comments: result: {}", res2);
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("delete(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
		if (res > 0)
		{
			oldInstance.flagDeleted();
			MySQLCache.cachePlantSpecies.remove(this.id);
		}
		oldInstance = null;
LOGGER.traceExit(log4jEntryMsg);
	}	// delete()

	private void doUpdate() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("doUpdate(): newInstance={}, somethingChanged={}", newInstance, somethingChanged);

		if (newInstance) return;
		if (!somethingChanged) return;
		StringBuilder query = new StringBuilder("update plantspecies set ");

		if (changedCropRotationGroupId)
		{
			query.append("cropRotationGroupId = ?, ");
		}

		if (changedCommonName)
		{
			query.append("commonName = ?, ");
		}

		if (changedLatinName)
		{
			query.append("latinName = ?, ");
		}

		if (changedDescription)
		{
			query.append("description = ?, ");
		}

		if (changedUtility)
		{
			query.append("utility = ?, ");
		}

		if (changedHardiness)
		{
			query.append("hardiness = ?, ");
		}

		if (changedLifeType)
		{
			query.append("lifeType = ?, ");
		}

		if (changedPlantType)
		{
			query.append("plantType = ?, ");
		}

		query.delete(query.length()-2, query.length());
		query.append(" where plantSpeciesId = ").append(this.id);
LOGGER.debug("doUpdate(): query={} ", query.toString());
		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString());	)
		{
			int paramIx = 1;
			if (changedCropRotationGroupId)
			{
				if (this.cropRotationGroupId == null)
				{
					LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.INTEGER);
				}
				else
				{
					LOGGER.debug("doUpdate(): param {}={}", paramIx, this.cropRotationGroupId);
					stmt.setInt(paramIx++, this.cropRotationGroupId);
				}
			}

			if (changedCommonName)
			{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.commonName);
				stmt.setString(paramIx++, this.commonName);
			}

			if (changedLatinName)
			{
				if (this.latinName == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.latinName);
					stmt.setString(paramIx++, this.latinName);
				}
			}

			if (changedDescription)
			{
				if (this.description == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.description);
					stmt.setString(paramIx++, this.description);
				}
			}

			if (changedUtility)
			{
				if (this.utility == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.utility);
					stmt.setString(paramIx++, this.utility);
				}
			}

			if (changedHardiness)
			{
				if (this.hardiness == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.hardiness);
					stmt.setString(paramIx++, this.hardiness);
				}
			}

			if (changedLifeType)
			{
				if (this.lifeType == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.lifeType);
					stmt.setString(paramIx++, this.lifeType);
				}
			}

			if (changedPlantType)
			{
				if (this.plantType == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.plantType);
					stmt.setString(paramIx++, this.plantType);
				}
			}

			stmt.executeUpdate();
//            conn.commit();

		}catch (SQLException ex) {
			LOGGER.error("doUpdate(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
LOGGER.traceExit(log4jEntryMsg);
	}	// doUpdate

	private void doInsert() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("doInsert(): newInstance={}", newInstance);

		if (!newInstance) return;
		if (!canSave())
		{
			throw LOGGER.throwing(Level.ERROR, new IllegalStateException("doInsert(): save criteria not met"));
		}
		if (!this.changedCommonName)
		{
			throw LOGGER.throwing(Level.ERROR, new IllegalStateException("PlantSpeciesBuilder: doInsert(): commonName unspecified"));
		}

		StringBuilder query = new StringBuilder("insert into plantspecies (");
		query.append("commonName, ");
		if (changedCropRotationGroupId)
		{
			query.append("cropRotationGroupId, ");
		}

		if (changedLatinName)
		{
			query.append("latinName, ");
		}

		if (changedDescription)
		{
			query.append("description, ");
		}

		if (changedUtility)
		{
			query.append("utility, ");
		}

		if (changedHardiness)
		{
			query.append("hardiness, ");
		}

		if (changedLifeType)
		{
			query.append("lifeType, ");
		}

		if (changedPlantType)
		{
			query.append("plantType, ");
		}

		query.replace(query.length()-2, query.length(), ") values (");
		query.append("?, ");
		if (changedCropRotationGroupId)
		{
			query.append("?, ");
		}

		if (changedLatinName)
		{
			query.append("?, ");
		}

		if (changedDescription)
		{
			query.append("?, ");
		}

		if (changedUtility)
		{
			query.append("?, ");
		}

		if (changedHardiness)
		{
			query.append("?, ");
		}

		if (changedLifeType)
		{
			query.append("?, ");
		}

		if (changedPlantType)
		{
			query.append("?, ");
		}

		query.replace(query.length()-2, query.length(), ")");
LOGGER.debug("doInsert(): query={}", query.toString());

		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString(), Statement.RETURN_GENERATED_KEYS);
            )
		{
			int paramIx = 1;
LOGGER.debug("doInsert(): param {}={}", paramIx, this.commonName);
			stmt.setString(paramIx++, this.commonName);
			if (changedCropRotationGroupId) {
				if (this.cropRotationGroupId == null)
				{
					LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.INTEGER);
				}
				else
				{
					LOGGER.debug("doInsert(): param {}={}", paramIx, this.cropRotationGroupId);
					stmt.setInt(paramIx++, this.cropRotationGroupId);
				}
			}

			if (changedLatinName) {
				if (this.latinName == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.latinName);
					stmt.setString(paramIx++, this.latinName);
				}
			}

			if (changedDescription) {
				if (this.description == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.description);
					stmt.setString(paramIx++, this.description);
				}
			}

			if (changedUtility) {
				if (this.utility == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.utility);
					stmt.setString(paramIx++, this.utility);
				}
			}

			if (changedHardiness) {
				if (this.hardiness == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.hardiness);
					stmt.setString(paramIx++, this.hardiness);
				}
			}

			if (changedLifeType) {
				if (this.lifeType == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.lifeType);
					stmt.setString(paramIx++, this.lifeType);
				}
			}

			if (changedPlantType) {
				if (this.plantType == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.plantType);
					stmt.setString(paramIx++, this.plantType);
				}
			}

			stmt.executeUpdate();

			ResultSet rs = stmt.getGeneratedKeys();
			rs.next();
			int newId = rs.getInt(1);
LOGGER.debug("doInsert(): newId: {}", newId);
			this.id = newId;
		}catch (SQLException ex) {
			LOGGER.error("doInsert(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

LOGGER.traceExit(log4jEntryMsg);
	}	// doInsert

	/**
     * Process the whole JSON array from a DUMP
     * 
     *  @param newVal    a list of JSON objects representing PlantSpecies as output by a JSON DUMP
     * 
     *	@throws	GNDBException	If the underlying MySQL database throws SQLException it is translated to this.
     *				The causal SQLException can be retrieved by <code>getCause()</code>
     * 
     * @since 2.2.5
     */
    void restoreJsonDump(List<JsonObject> newVal) throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("restoreJsonDump(list JSON)");
        
        if (newVal.isEmpty())
            return;

		StringBuilder query = new StringBuilder("insert into plantspecies (");
		query.append("plantSpeciesId, ");
		query.append("cropRotationGroupId, ");
		query.append("commonName, ");
		query.append("latinName, ");
		query.append("description, ");
		query.append("utility, ");
		query.append("hardiness, ");
		query.append("lifeType, ");
		query.append("plantType, ");
		query.append("lastUpdated, ");
		query.append("created ");
		query.append(") ");
        if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.hsqldb )
        {
            query.append(" overriding system value ");
        }
        query.append(" values (");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("? ");
		query.append(")");
LOGGER.debug("doJsonInsert(): query={}", query.toString());
        
		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString());	)
		{
            conn.setAutoCommit(false);
            int txCount = 0;
            if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
            {
                conn.createStatement().execute("SET IDENTITY_INSERT plantspecies ON");
            }
            
            for (JsonObject jo : newVal)
            {
                if (!"DUMP".equals(jo.getString("JsonMode", "DUMP")))
                {
                    LOGGER.error("PlantSpecies DUMP object is not DUMP");
                    throw new IllegalArgumentException("PlantSpecies DUMP object is not DUMP");
                }
                if (!"PlantSpecies".equals(jo.getString("JsonNBClass", "PlantSpecies")))
                {
                    LOGGER.error("PlantSpecies DUMP object is not PlantSpecies");
                    throw new IllegalArgumentException("PlantSpecies DUMP object is not PlantSpecies");
                }
                PlantSpecies ps = new PlantSpecies(jo);
                if (ps.getId() <= 0)
                {//this forces the value of the id field.  The >0 test is a bodge.
                    LOGGER.error("PlantSpecies DUMP object does not have an id");
                    throw new IllegalArgumentException("PlantSpecies DUMP object does not have an id");
                }
                
                int paramIx = 1;
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getId());
                stmt.setInt(paramIx++, ps.getId());

				if (ps.getCropRotationGroupId() == null)
				{
					LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
					stmt.setNull(paramIx++, Types.INTEGER);
				}
				else
				{
					LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getCropRotationGroupId());
					stmt.setInt(paramIx++, ps.getCropRotationGroupId());
				}

				LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getCommonName());
                stmt.setString(paramIx++, ps.getCommonName());

                if (ps.getLatinName().isEmpty())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getLatinName().get());
                    stmt.setString(paramIx++, ps.getLatinName().get());
                }

                if (ps.getDescription().isEmpty())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getDescription().get());
                    stmt.setString(paramIx++, ps.getDescription().get());
                }

                if (ps.getUtility().isEmpty())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getUtility().get());
                    stmt.setString(paramIx++, ps.getUtility().get());
                }

                if (ps.getHardiness().isEmpty())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getHardiness().get());
                    stmt.setString(paramIx++, ps.getHardiness().get());
                }

                if (ps.getLifeType().isEmpty())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getLifeType().get());
                    stmt.setString(paramIx++, ps.getLifeType().get());
                }

                if (ps.getPlantType().isEmpty())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getPlantType().get());
                    stmt.setString(paramIx++, ps.getPlantType().get());
                }

LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getLastUpdated());
                stmt.setTimestamp(paramIx++, Timestamp.valueOf(ps.getLastUpdated()));
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getCreated());
                stmt.setTimestamp(paramIx++, Timestamp.valueOf(ps.getCreated()));

                stmt.executeUpdate();

                if (!ps.getComments().isEmpty())
                {
                    if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
                    {
                        conn.createStatement().execute("SET IDENTITY_INSERT plantspecies OFF");
                    }
                    CommentBuilder cb = new CommentBuilder(NotebookEntryType.PLANTSPECIES, ps.getId());
                    cb.doJsonInsert(ps.getComments(), conn);
                    if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
                    {
                        conn.createStatement().execute("SET IDENTITY_INSERT plantspecies ON");
                    }
                }

                if (++txCount > 50)
                {
                    conn.commit();
                    txCount = 0;
                }
            }
            conn.commit();
            if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
            {
                conn.createStatement().execute("SET IDENTITY_INSERT plantspecies OFF");
            }
		} catch (SQLException ex) {
			LOGGER.error("restoreJsonDump(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

LOGGER.traceExit(log4jEntryMsg);
    }	// restoreJsonDump(JsonObjects)

    
	@Override
	public boolean hasAncestor() throws GNDBException
	{
		//	this class does not participate in story lines
		return false;
	}	//	hasAncestor()

	@Override
	public StoryLineTree<? extends INotebookEntry> getAncestors() throws GNDBException
	{
		//	this class does not participate in story lines
		return StoryLineTree.emptyTree();
	}	//	getAncestors()

	@Override
	public IPlantSpeciesBuilder ancestor(IPurchaseItem ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(PurchaseItem)

	@Override
	public IPlantSpeciesBuilder ancestor(IGroundwork ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(Groundwork)

	@Override
	public IPlantSpeciesBuilder ancestor(IAfflictionEvent ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(AfflictionEvent)

	@Override
	public IPlantSpeciesBuilder ancestor(IHusbandry ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(Husbandry)

	@Override
	public IPlantSpeciesBuilder ancestor(ISaleItem ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(SaleItem)

	@Override
	public boolean hasDescendant() throws GNDBException
	{
		//	this class does not participate in story lines
		return false;
	}	//	hasDescendant()

	@Override
	public StoryLineTree<? extends INotebookEntry> getDescendants() throws GNDBException
	{
		//	this class does not participate in story lines
		return StoryLineTree.emptyTree();
	}	//	getDescendants()

	@Override
	public boolean addDescendant(IPurchaseItem descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(PurchaseItem)

	@Override
	public boolean addDescendant(IGroundwork descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(Groundwork)

	@Override
	public boolean addDescendant(IAfflictionEvent descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(AfflictionEvent)

	@Override
	public boolean addDescendant(IHusbandry descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(Husbandry)

	@Override
	public boolean addDescendant(ISaleItem descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(SaleItem)

}

