/*
 * Copyright (C) 2018, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Use FlagHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import java.beans.PropertyChangeListener;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDateTime;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;

//import java.time.*;
import java.util.Optional;

import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonArray;
import jakarta.json.JsonObjectBuilder;
import jakarta.json.JsonObject;

/**
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

final class Product implements IProduct
{
	private final FlagHandler<IProduct> flagHandler;
	{
		flagHandler = new FlagHandler<>(this);
	}

	private final int id;
	private final int productCategoryId;

	/*
	*	The product is plants of this species.
@apiNote
If present, plantVarietyId may or may not be given.  This 'denormalises' the model but enables easy searches for e.g. 'all tomatoes'
	*/
	private final Integer plantSpeciesId;

	/*
	*	The product is plants of this species.
@apiNote
If present, plantSpeciesId must be given.  This 'denormalises' the model but enables easy searches for e.g. 'all tomatoes'
	*/
	private final Integer plantVarietyId;
	private final Integer productBrandId;

	/*
	*	For plant like products (seeds, etc.) the plant species common name.<BR>
<B>This is unreliable - editing the plant species might not change the value here!</B>
	*/
	private final String name;

	/*
	*	For plant-like products (seeds, etc.) the plant variety (if present) common name<BR>
<B>This is unreliable - editing the plant variety might not change the value here!</B>
	*/
	private final String nameDetail_1;
	private final String nameDetail_2;
	private final String description;
	private final LocalDateTime lastUpdated;
	private final LocalDateTime created;
	private final List<Comment> commentsList;

	/**
	*	Build an immutable Product entry one field at a time
	*/
	Product(
		final int id,
		final int productCategoryId,
		final Integer plantSpeciesId,
		final Integer plantVarietyId,
		final Integer productBrandId,
		final String name,
		final String nameDetail_1,
		final String nameDetail_2,
		final String description,
		final LocalDateTime lastUpdated,
		final LocalDateTime created,
		final Comment... comments)
	{
		this.id = id;
		this.productCategoryId = productCategoryId;
		this.plantSpeciesId = plantSpeciesId;
		this.plantVarietyId = plantVarietyId;
		this.productBrandId = productBrandId;
		this.name = name;
		this.nameDetail_1 = nameDetail_1;
		this.nameDetail_2 = nameDetail_2;
		this.description = description;
		this.lastUpdated = lastUpdated;
		this.created = created;
		if (comments != null && comments.length>0)
		{
			this.commentsList = new ArrayList<>(Arrays.asList(comments));
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}

	/**
	*	Build an immutable Product entry cloning the given Product entry but adding the comments list
	*/
	Product(
		final Product toCopy,
		final Comment... comments)
	{
		this(toCopy, Arrays.asList(comments));
	}

	/**
	*	Build an immutable Product entry cloning the given Product entry but adding the comments list
	*/
	Product(
		final Product toCopy,
		final List<Comment> comments)
	{
		this.id = toCopy.id;
		this.productCategoryId = toCopy.productCategoryId;
		this.plantSpeciesId = toCopy.plantSpeciesId;
		this.plantVarietyId = toCopy.plantVarietyId;
		this.productBrandId = toCopy.productBrandId;
		this.name = toCopy.name;
		this.nameDetail_1 = toCopy.nameDetail_1;
		this.nameDetail_2 = toCopy.nameDetail_2;
		this.description = toCopy.description;
		this.lastUpdated = toCopy.lastUpdated;
		this.created = toCopy.created;
		if (comments != null && comments.size()>0)
		{
			if (toCopy.commentsList.size()>0)
			{
				// append new comments to previous list
				this.commentsList = new ArrayList<>(toCopy.commentsList);
				this.commentsList.addAll(comments);
			}
			else
			{	// no comments on original item
				this.commentsList = new ArrayList<>(comments);
			}
		}
		else
		{	// no new comments to add
			this.commentsList = toCopy.commentsList;
		}
	}

	/**
	*	Build an immutable Product entry from a JSON dump.
	*	The dumped object must be complete (all non-nullable fields must have values) except
	*	the id field can be null or absent to indicate that this is a new item to be inserted.
	*
	*	@param	json	a JsonObject holding all the fields for a full initialisation.
	*/
	Product(JsonObject json)
	{
		this.id = json.getInt("id", -1);
		this.productCategoryId = json.getInt("productCategoryId");
		if (json.containsKey("plantSpeciesId") && !json.isNull("plantSpeciesId"))
		{
			this.plantSpeciesId = json.getInt("plantSpeciesId");
		}
		else
		{
			this.plantSpeciesId = null;
		}

		if (json.containsKey("plantVarietyId") && !json.isNull("plantVarietyId"))
		{
			this.plantVarietyId = json.getInt("plantVarietyId");
		}
		else
		{
			this.plantVarietyId = null;
		}

		if (json.containsKey("productBrandId") && !json.isNull("productBrandId"))
		{
			this.productBrandId = json.getInt("productBrandId");
		}
		else
		{
			this.productBrandId = null;
		}

		this.name = json.getString("name");
		if (json.containsKey("nameDetail_1") && !json.isNull("nameDetail_1"))
		{
			this.nameDetail_1 = json.getString("nameDetail_1");
		}
		else
		{
			this.nameDetail_1 = null;
		}

		if (json.containsKey("nameDetail_2") && !json.isNull("nameDetail_2"))
		{
			this.nameDetail_2 = json.getString("nameDetail_2");
		}
		else
		{
			this.nameDetail_2 = null;
		}

		if (json.containsKey("description") && !json.isNull("description"))
		{
			this.description = json.getString("description");
		}
		else
		{
			this.description = null;
		}

		this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
		this.created = LocalDateTime.parse(json.getString("created"));
		JsonArray jsonComments = json.getJsonArray("comments");
		if (jsonComments != null && !jsonComments.isEmpty())
		{// there is probably only one comment
			this.commentsList = new ArrayList<>(jsonComments.size());
			for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
			{
				this.commentsList.add(new Comment(ct));
			}
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}	//constructor from JSON

	int getId()
	{
		return id;
	}
	@Override
	public Integer getKey()
	{
		return id;
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.PRODUCT;
	}

	int getProductCategoryId()
	{
		return productCategoryId;
	}
	@Override
	public IProductCategory getProductCategory()
	{
		return MySQLCache.cacheProductCategory.get(productCategoryId);
	}

	Integer getPlantSpeciesId()
	{
		return plantSpeciesId;
	}
	@Override
	public Optional<IPlantSpecies> getPlantSpecies()
	{
		return Optional.ofNullable(MySQLCache.cachePlantSpecies.get(plantSpeciesId));
	}

	Integer getPlantVarietyId()
	{
		return plantVarietyId;
	}
	@Override
	public Optional<IPlantVariety> getPlantVariety()
	{
		return Optional.ofNullable(MySQLCache.cachePlantVariety.get(plantVarietyId));
	}

	Integer getProductBrandId()
	{
		return productBrandId;
	}
	@Override
	public Optional<IProductBrand> getProductBrand()
	{
		return Optional.ofNullable(MySQLCache.cacheProductBrand.get(productBrandId));
	}

	@Override
	public String getName()
	{
		return name;
	}

	@Override
	public Optional<String> getNameDetail_1()
	{
		return Optional.ofNullable(nameDetail_1);
	}

	@Override
	public Optional<String> getNameDetail_2()
	{
		return Optional.ofNullable(nameDetail_2);
	}

	@Override
	public Optional<String> getDescription()
	{
		return Optional.ofNullable(description);
	}

	@Override
	public LocalDateTime getLastUpdated()
	{
		return lastUpdated;
	}

	@Override
	public LocalDateTime getCreated()
	{
		return created;
	}

	@Override
	public boolean sameAs(INotebookEntry other)
	{
		if (other == null || other.getType() != this.getType())
		{
			return false;
		}
		return other.getKey().equals(this.getKey());
	}

	@Override
	public IPurchaseItemLister getPurchaseItem()
	{
		return new PurchaseItemLister().product(this);
	}

	@Override
	public IRetailerHasProductLister getRetailerHasProduct()
	{
		return new RetailerHasProductLister().product(this);
	}

	@Override
	public IShoppingListLister getShoppingList()
	{
		return new ShoppingListLister().product(this);
	}

	@Override
	public List<IComment> getComments() {
		return new ArrayList<>(this.commentsList);
	}

	JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
	{
		JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
		jsonBuilder.add("id", id);
		jsonBuilder.add("productCategoryId", productCategoryId);
		if (plantSpeciesId != null)
		{
			jsonBuilder.add("plantSpeciesId", plantSpeciesId);
		}
		else
		{
			jsonBuilder.addNull("plantSpeciesId");
		}
		if (plantVarietyId != null)
		{
			jsonBuilder.add("plantVarietyId", plantVarietyId);
		}
		else
		{
			jsonBuilder.addNull("plantVarietyId");
		}
		if (productBrandId != null)
		{
			jsonBuilder.add("productBrandId", productBrandId);
		}
		else
		{
			jsonBuilder.addNull("productBrandId");
		}
		jsonBuilder.add("name", name);
		if (nameDetail_1 != null)
		{
			jsonBuilder.add("nameDetail_1", nameDetail_1);
		}
		else
		{
			jsonBuilder.addNull("nameDetail_1");
		}
		if (nameDetail_2 != null)
		{
			jsonBuilder.add("nameDetail_2", nameDetail_2);
		}
		else
		{
			jsonBuilder.addNull("nameDetail_2");
		}
		if (description != null)
		{
			jsonBuilder.add("description", description);
		}
		else
		{
			jsonBuilder.addNull("description");
		}
		jsonBuilder.add("lastUpdated", lastUpdated.toString());
		jsonBuilder.add("created", created.toString());
		if (commentsList != null && !commentsList.isEmpty())
		{// no point writing an empty comments array (the loaders handle this)
			JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
			for (Comment ct : commentsList)
			{
				jsonComments.add(ct.toJson(jsonFactory));
			}
			jsonBuilder.add("comments", jsonComments);
		}
		jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "Product");
		return jsonBuilder;
	}	//	toJson

	@Override
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.addPropertyChangeListener(propertyName, listener);
	}

	@Override
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.removePropertyChangeListener(propertyName, listener);
	}

	@Override
	public void flagDeleted()
	{
		flagHandler.flagDeleted();
	}

	@Override
	public void flagReplaced(final IProduct newValue)
	{
		flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
	}

	@Override
	public void flagChildDeleted(final IPurchaseItem child)
	{
		flagHandler.flagChildDeleted("PurchaseItem", child);
	}

	@Override
	public void flagChildAdded(final IPurchaseItem child)
	{
		flagHandler.flagChildAdded("PurchaseItem", child);
	}

	@Override
	public void flagChildDeleted(final IRetailerHasProduct child) {
		flagHandler.flagChildDeleted("RetailerHasProduct", child);
	}

	@Override
	public void flagChildAdded(final IRetailerHasProduct child) {
		flagHandler.flagChildAdded("RetailerHasProduct", child);
	}

	@Override
	public void flagChildDeleted(final IShoppingList child)
	{
		flagHandler.flagChildDeleted("ShoppingList", child);
	}

	@Override
	public void flagChildAdded(final IShoppingList child)
	{
		flagHandler.flagChildAdded("ShoppingList", child);
	}


	@Override
	public String toString() {
		return "Product: " + "id: " + id + ", " +
				"productCategoryId: " + productCategoryId + ", " +
				"plantSpeciesId: " + plantSpeciesId + ", " +
				"plantVarietyId: " + plantVarietyId + ", " +
				"productBrandId: " + productBrandId + ", " +
				"name: " + name + ", " +
				"nameDetail_1: " + nameDetail_1 + ", " +
				"nameDetail_2: " + nameDetail_2 + ", " +
				"description: " + description + ", " +
				"lastUpdated: " + lastUpdated + ", " +
				"created: " + created;
	}

}
