/*
 * Copyright (C) 2018, 2019, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.util.List;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import java.time.LocalDateTime;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonObjectBuilder;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;

/**
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

final class ProductCategoryLister implements IProductCategoryLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private boolean useName = false;
	private String[] nameList = new String[10];
	private int nameNext = 0;	// next free slot in list

	private boolean useId = false;
	private int[] idList = new int[10];
	private int idNext = 0;	// next free slot in list

	ProductCategoryLister() {}

	@Override
	public List<IProductCategory> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		if (MySQLCache.invalidProductCategory)
		{
			load();
		}

		if(!useId && !useName)
		{
			useId = false;
			idNext = 0;
			useName = false;
			nameNext = 0;
			MySQLCache.invalidProductCategory = false;
			return MySQLCache.cacheProductCategory.values().stream().
					sorted((a, b) -> a.getName().compareToIgnoreCase(b.getName())).collect(Collectors.toList());
		}

		List<IProductCategory> outList = new ArrayList<>();

		idList = Arrays.copyOf(idList, idNext);

		if (useName)
		{
			nameList = Arrays.copyOf(nameList, nameNext);
			int [] nameKeys = MySQLCache.cacheProductCategory.values().stream().
				filter(c -> Arrays.stream(nameList).anyMatch(p -> p.equals(c.getName())))
				.mapToInt(INotebookEntry::getKey).
				toArray();
			idList = IntStream.concat(Arrays.stream(nameKeys), Arrays.stream(idList)).toArray();
		}

		idList = Arrays.stream(idList).distinct().toArray();

		for (int i : idList)
		{
			outList.add(MySQLCache.cacheProductCategory.get(i));

		}
		useId = false;
		idNext = 0;
		useName = false;
		nameNext = 0;
		MySQLCache.invalidProductCategory = false;

LOGGER.traceExit(log4jEntryMsg);
		return outList.stream().sorted((a,b) -> a.getName().compareToIgnoreCase(b.getName())).collect(Collectors.toList());
	}	// fetch()

	void load() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("load()");

		if (MySQLCache.invalidProductCategory)
		{
			List<ProductCategory> tempList = new ArrayList<>();

			String query = "";
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = buildSQL_MySQL();
				case hsqldb -> query = buildSQL_hsqldb();
				case MSSQLServer -> query = buildSQL_MSSQLServer();
				default -> {
					LOGGER.debug("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("no known RDBMS"));
				}
			}
LOGGER.debug("load(): query: {}", query);
			try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
			{
				ResultSet rs = stmt.executeQuery(query);
				switch (DBConnection.DB_IN_USE)
				{
					case MariaDB, MySQL -> tempList = processResults_MySQL(rs);
					case hsqldb -> tempList = processResults_hsqldb(rs);
					case MSSQLServer -> tempList = processResults_MSSQLServer(rs);
					default -> {
						LOGGER.debug("fetch(): no known rdbms");
						throw new GNDBException(new IllegalStateException("no known RDBMS"));
					}
				}
				stmt.close();
			}catch (SQLException ex) {
				LOGGER.error("load(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
				throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
			}
			for (ProductCategory ps : tempList)
			{
				MySQLCache.cacheProductCategory.putIfAbsent(ps.getId(), ps);
			}
		}

LOGGER.traceExit(log4jEntryMsg);
	}	// load()

    private String buildSQL_MySQL()
    {
		return "select d.*, c.* from productcategory as d " + "left join (select * from comment where ownerType = \"PC\") as c " +
				"on d.productCategoryId = c.ownerId " +
				" order by LOWER(d.name), c.date";
    }

    private String buildSQL_hsqldb()
    {
		return "select d.*, c.* from productcategory as d " + "left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'PC') as c " +
				"on d.productCategoryId = c_ownerId " +
				" order by LOWER(d.name), c_date";
    }

    private String buildSQL_MSSQLServer()
    {
		return "select d.productCategoryId as d_productCategoryId, " +
				"d.name as d_name, " +
				"d.description as d_description, " +
				"d.plantLike as d_plantLike, " +
				"d.created as d_created, " +
				"d.lastUpdated as d_lastUpdated," +
				" c.* from productcategory as d " + "left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'PC') as c " +
				"on d.productCategoryId = c_ownerId " +
				" order by LOWER(d.name), c_date";
    }

	private List<ProductCategory> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<ProductCategory> tempList = new ArrayList<>();

		ProductCategory item = null;

		while (rs.next())
		{
			int productCategoryId = rs.getInt("d.productCategoryId");
			String name = rs.getString("d.name");
			String description = rs.getString("d.description");
			boolean plantLike = rs.getBoolean("d.plantLike");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("processResults_MySQL(): productCategoryId: {}, name: {}, description: {}, plantLike: {}, lastUpdated: {}, created: {}",
                                productCategoryId, name, description, plantLike, lastUpdated, created);
			if (item != null && productCategoryId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c.commentId"),
					rs.getInt("c.ownerId"),
					"PC",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new ProductCategory(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new ProductCategory(productCategoryId, name, description, plantLike, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						productCategoryId,
						"PC",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new ProductCategory(productCategoryId, name, description, plantLike, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<ProductCategory> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<ProductCategory> tempList = new ArrayList<>();

		ProductCategory item = null;

		while (rs.next())
		{
			int productCategoryId = rs.getInt("productcategory.productCategoryId");
			String name = rs.getString("productcategory.name");
			String description = rs.getString("productcategory.description");
			boolean plantLike = rs.getBoolean("productcategory.plantLike");
			LocalDateTime lastUpdated = rs.getTimestamp("productcategory.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("productcategory.created").toLocalDateTime();
LOGGER.debug("processResults_hsqldb(): productCategoryId: {}, name: {}, description: {}, plantLike: {}, lastUpdated: {}, created: {}",
                                productCategoryId, name, description, plantLike, lastUpdated, created);
			if (item != null && productCategoryId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"PC",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new ProductCategory(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new ProductCategory(productCategoryId, name, description, plantLike, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						productCategoryId,
						"PC",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new ProductCategory(productCategoryId, name, description, plantLike, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<ProductCategory> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<ProductCategory> tempList = new ArrayList<>();

		ProductCategory item = null;

		while (rs.next())
		{
			int productCategoryId = rs.getInt("d_productCategoryId");
			String name = rs.getString("d_name");
			String description = rs.getString("d_description");
			boolean plantLike = rs.getBoolean("d_plantLike");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("processResults_hsqldb(): productCategoryId: {}, name: {}, description: {}, plantLike: {}, lastUpdated: {}, created: {}",
                                productCategoryId, name, description, plantLike, lastUpdated, created);
			if (item != null && productCategoryId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"PC",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new ProductCategory(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new ProductCategory(productCategoryId, name, description, plantLike, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						productCategoryId,
						"PC",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new ProductCategory(productCategoryId, name, description, plantLike, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	void clear()
	{
		MySQLCache.cacheProductCategory.clear();
		MySQLCache.invalidProductCategory = true;
		MySQLCache.completeProductCategory = false;
	}

	/**
	*
	*Select only the ProductCategory entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	ProductCategoryLister id(int... vals)
	{
		useId = true;
		if (idNext + vals.length >= idList.length)
		{
			idList = Arrays.copyOf(idList, idList.length+vals.length+10);
		}
		for (int val : vals)
		{
			idList[idNext++] = val;
		}
		return this;
	}

	@Override
	public IProductCategoryLister name(String... vals)
	{
		if (vals == null) return this;
		if (vals.length == 0) return this;
		useName = true;
		if (nameNext + vals.length >= nameList.length)
		{
			nameList = Arrays.copyOf(nameList, nameList.length + vals.length + 10);
		}
		for (String val : vals) {nameList[nameNext++] = val;}
		return this;
	}

	@Override
	public IProductCategoryLister key(IProductCategory... vals)
	{
		if (vals == null) return this;
		if (vals.length == 0) return this;
		int[] keys = new int[vals.length];
		int keyCount = 0;
		for (IProductCategory val : vals) {
			if (val == null) continue;
			Integer ky = val.getKey();
			if (ky == null) continue;
			keys[keyCount++] = ky;
		}
		if (keyCount == 0) return this;
		keys = Arrays.copyOf(keys, keyCount);	// trim array to actual size - should be a null-op
		return this.id(keys);
	}

	@Override
	public IProductCategoryLister key(List<IProductCategory> items)
	{
		if (items == null) return this;
		if (items.isEmpty()) return this;
		return this.key(items.toArray(new IProductCategory[0]));
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("toJson()");

		if (MySQLCache.invalidProductCategory)
		{
			load();
		}

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IProductCategory ihc : MySQLCache.cacheProductCategory.values())
		{
			LOGGER.debug("PC: {}", ihc);
			ProductCategory hc = (ProductCategory)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "ProductCategory");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "ProductCategory.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJSON(): IOException: ", ex);
		}
	}	// toJson

}
