/*
 *
 *  Copyright (C) 2021, 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
    Change log
	3.1.0	Use jakarta implementation of JSON
*/

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import jakarta.json.*;

import java.beans.PropertyChangeListener;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Year;
import java.util.*;

/**
 *{@inheritDoc}
 *
 *	@author	Andy Gegg
 *	@version	3.1.0
 *	@since	3.0.0
 */

final class Review implements IReview
{
    private final FlagHandler<IReview> flagHandler;
    {
        flagHandler = new FlagHandler<>(this);
    }

    private final int id;

    /*
    *   When the review was written
     */
    private final LocalDate date;

    /*
    *   The year covered by the review
    */
    private final Year yearInReview;

    /*
    *   The period covered
     */
    private final LocalDate coverFrom;
    private final LocalDate coverTo;

    /*
    *   Title for the review
     */
    private final String title;

    /*
    *   The text of the review
     */
    private final String description;

    private final LocalDateTime lastUpdated;
    private final LocalDateTime created;
    private final List<Comment> commentsList;

    /**
     *	Build an immutable Review entry one field at a time
     */
    Review(
            final int id,
            final LocalDate date,
            Year yearInReview,
            LocalDate coverFrom,
            LocalDate coverTo,
            final String title,
            final String description,
            final LocalDateTime lastUpdated,
            final LocalDateTime created,
            final Comment... comments)
    {
        this.id = id;
        this.date = date;
        this.yearInReview = yearInReview;
        this.coverFrom = coverFrom;
        this.coverTo = coverTo;
        this.title = title;
        this.description = description;
        this.lastUpdated = lastUpdated;
        this.created = created;

        if (comments != null && comments.length>0)
        {
            this.commentsList = new ArrayList<>(Arrays.asList(comments));
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }

    /**
     *	Build an immutable Journal entry cloning the given Journal entry but adding the comments list
     */
    Review(
            final Review toCopy,
            final Comment... comments)
    {
        this(toCopy, Arrays.asList(comments));
    }

    /**
     *	Build an immutable Review entry cloning the given Review entry but adding the comments list
     */
    Review(
            final Review toCopy,
            final List<Comment> comments)
    {
        this.id = toCopy.id;
        this.date = toCopy.date;
        this.yearInReview = toCopy.yearInReview;
        this.coverFrom = toCopy.coverFrom;
        this.coverTo = toCopy.coverTo;
        this.title = toCopy.title;
        this.description = toCopy.description;
        this.lastUpdated = toCopy.lastUpdated;
        this.created = toCopy.created;

        if (comments != null && comments.size()>0)
        {
            if (toCopy.commentsList.size()>0)
            {
                // append new comments to previous list
                this.commentsList = new ArrayList<>(toCopy.commentsList);
                this.commentsList.addAll(comments);
            }
            else
            {	// no comments on original item
                this.commentsList = new ArrayList<>(comments);
            }
        }
        else
        {	// no new comments to add
            this.commentsList = toCopy.commentsList;
        }
    }

    /**
     *	Build an immutable Review entry from a JSON dump.
     *	The dumped object must be complete (all non-nullable fields must have values) except
     *	the id field can be null or absent to indicate that this is a new item to be inserted.
     *
     *	@param	json	a JsonObject holding all the fields for a full initialisation.
     */
    Review(JsonObject json)
    {
        this.id = json.getInt("id", -1);

        this.date = LocalDate.parse(json.getString("date"));
        this.yearInReview = Year.parse(json.getString("yearInReview"));

        if (json.containsKey("coverFrom") && !json.isNull("coverFrom"))
        {
            this.coverFrom = LocalDate.parse(json.getString("coverFrom"));
        }
        else
        {
            this.coverFrom = null;
        }
        if (json.containsKey("coverTo") && !json.isNull("coverTo"))
        {
            this.coverTo = LocalDate.parse(json.getString("coverTo"));
        }
        else
        {
            this.coverTo = null;
        }

        this.title = json.getString("title");

        if (json.containsKey("description") && !json.isNull("description"))
        {
            this.description = json.getString("description");
        }
        else
        {
            this.description = null;
        }

        this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
        this.created = LocalDateTime.parse(json.getString("created"));

        JsonArray jsonComments = json.getJsonArray("comments");
        if (jsonComments != null && !jsonComments.isEmpty())
        {// there is probably only one comment
            this.commentsList = new ArrayList<>(jsonComments.size());
            for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
            {
                this.commentsList.add(new Comment(ct));
            }
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }	//constructor from JSON

    int getId()
    {
        return id;
    }
    @Override
    public Integer getKey()
    {
        return id;
    }

    @Override
    public NotebookEntryType getType()
    {
        return NotebookEntryType.REVIEW;
    }

    @Override
    public LocalDate getDate()
    {
        return date;
    }

    @Override
    public Year getYearInReview()
    {
        return yearInReview;
    }

    @Override
    public Optional<LocalDate> getCoverFrom()
    {
        return Optional.ofNullable(coverFrom);
    }

    @Override
    public Optional<LocalDate> getCoverTo()
    {
        return Optional.ofNullable(coverTo);
    }

    @Override
    public String getTitle()
    {
        return title;
    }

    @Override
    public Optional<String> getDescription()
    {
        return Optional.ofNullable(description);
    }

    @Override
    public LocalDateTime getLastUpdated()
    {
        return lastUpdated;
    }

    @Override
    public LocalDateTime getCreated()
    {
        return created;
    }

    @Override
    public boolean sameAs(INotebookEntry other)
    {
        if (other == null || other.getType() != this.getType())
        {
            return false;
        }
        return other.getKey().equals(this.getKey());
    }

    @Override
    public List<IComment> getComments()
    {
        return new ArrayList<>(this.commentsList);
    }

    JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
    {
        JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
        jsonBuilder.add("id", id);

        jsonBuilder.add("date", date.toString());
        jsonBuilder.add("yearInReview", yearInReview.toString());

        if (coverFrom != null)
        {
            jsonBuilder.add("coverFrom", coverFrom.toString());
        }
        else
        {
            jsonBuilder.addNull("coverFrom");
        }
        if (coverTo != null)
        {
            jsonBuilder.add("coverTo", coverTo.toString());
        }
        else
        {
            jsonBuilder.addNull("coverTo");
        }

        jsonBuilder.add("title", title);
        if (description != null)
        {
            jsonBuilder.add("description", description);
        }
        else
        {
            jsonBuilder.addNull("description");
        }

        jsonBuilder.add("lastUpdated", lastUpdated.toString());
        jsonBuilder.add("created", created.toString());
        if (commentsList != null && !commentsList.isEmpty())
        {// no point writing an empty comments array (the loaders handle this)
            JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
            for (Comment ct : commentsList)
            {
                jsonComments.add(ct.toJson(jsonFactory));
            }
            jsonBuilder.add("comments", jsonComments);
        }
        jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "Review");
        return jsonBuilder;
    }	//	toJson

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.removePropertyChangeListener(propertyName, listener);
    }

    @Override
    public void flagDeleted()
    {
        flagHandler.flagDeleted();
    }

    @Override
    public void flagReplaced(IReview newValue)
    {
        flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
    }

    @Override
    public String toString() {
        return "Review: " + "id: " + id + ", " +
                "date: " + date + ", " +
                "yearInReview: " + yearInReview + ", " +
                "coverFrom: " + coverFrom + ", " +
                "coverTo: " + coverTo + ", " +
                "title: " + title + ", " +
                "description: " + (description != null ? (description.length() > 256 ? description.substring(0, 256) : description) : null )+ ", " +
                "lastUpdated: " + lastUpdated + ", " +
                "created: " + created;
    }

}
