/*
 * Copyright (C) 2018-2020, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.8.1   Prepare to replace this as a record in Java 14
    2.9.1   Rewrite as a Record
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.NotebookEntryType;

import java.time.*;
//import javax.json.JsonBuilderFactory;
import jakarta.json.JsonObject;
//import javax.json.JsonObjectBuilder;


/**
*
*	This class is used ONLY within the database handler code.
*	Other databases may use a different mechanism to record story lines.
* 
*   Possible values for the ancestor and descendant types are:
<UL>
<LI>AE - affliction event
<LI>HU - husbandry
<LI>GW - groundwork
<LI>PU - purchase
<LI>PI - purchase item
</UL>
*
*This is the 'closure table' for the storyline (thread) hierarchy.<BR>
Given a tree structure<BR> 
{@code A->B->C, A->D}<BR>
there is an entry for each link, chased all the way back up:<BR>
(A,A,0) (A,B,1) (B,B,0) (A,C,2) (C,C,0) (B,C,1),<BR>
(A,D,1) (D,D,0)<BR>
It's easy to get the id of every ancestor or descendant of a given node.<BR>
The self-referencing nodes (A,A) make it easy to insert new nodes (single SQL rather than several bits).<BR>
Multiple parents are supported (cf a BoM structure).<BR>

For appropriate SQL see, e.g.,<BR>
https://www.slideshare.net/billkarwin/models-for-hierarchical-data
(slide 41 et seq)<BR>
and also<BR>
https://gist.github.com/ekillaby/2377806
and also<BR>
http://karwin.blogspot.co.uk/2010/03/rendering-trees-with-closure-tables.html

*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

record StoryLineIndex(int id,
        int ancestorId, // The database key of the ancestor
		String ancestorTypeString, // The type of the ancestor.
		int descendantId,  // The database key of the descendant
		String descendantTypeString,   // The type of the descendant.
		int depth, // How far down the tree the descendant node is compared to the ancestor
		LocalDateTime created)
{

	/*
	*	The type of the ancestor.
<UL>
<LI>AE - affliction event
<LI>HU - husbandry
<LI>GW - groundwork
<LI>PU - purchase
<LI>PI - purchase item
</UL>
	*/

	/*
	*	The type of the descendant.
<UL>
<LI>AE - affliction event
<LI>HU - husbandry
<LI>GW - groundwork
<LI>SA - sale
<LI>SI - sale item
</UL>
	*/

	/**
	*	Build an immutable StoryLineIndex entry from a JSON dump.
	*	The dumped object must be complete (all non-nullable fields must have values).
	*
	*	@param	json	a JsonObject holding all the fields for a full initialisation.
	*/
	StoryLineIndex(JsonObject json)
	{
        this(   json.getInt("id", -1),
                json.getInt("ancestorId"),
                json.getString("ancestorType"), //  NB this is the DB column name and is deliberately different from the name here
                json.getInt("descendantId"),
                json.getString("descendantType"),   //  NB this is the DB column name and is deliberately different from the name here
                json.getInt("depth"),
                LocalDateTime.parse(json.getString("created"))
            );
	}

	/**
	*	get the value of The type of the ancestor.
<UL>
<LI>AE - affliction event
<LI>HU - husbandry
<LI>GW - groundwork
<LI>PU - purchase
<LI>PI - purchase item
</UL>
	*/
	NotebookEntryType getAncestorType()
	{
        return NotebookEntryType.fromString(ancestorTypeString);
	}

	/**
	*	get the value of The type of the descendant.
<UL>
<LI>AE - affliction event
<LI>HU - husbandry
<LI>GW - groundwork
<LI>SA - sale
<LI>SI - sale item
</UL>
	*/
	NotebookEntryType getDescendantType()
	{
        return NotebookEntryType.fromString(descendantTypeString);
	}

//	JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
//	{
//		JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
//		jsonBuilder.add("id", id);
//		jsonBuilder.add("ancestorId", ancestorId);
//		jsonBuilder.add("ancestorType", ancestorTypeString);
//		jsonBuilder.add("descendantId", descendantId);
//		jsonBuilder.add("descendantType", descendantTypeString);
//		jsonBuilder.add("depth", depth);
//		jsonBuilder.add("created", created.toString());
//		jsonBuilder.add("JsonMode", "DUMP");
//        jsonBuilder.add("JsonNBClass", "StoryLineIndex");
// 		return jsonBuilder;
//	}


}
