/*
 * Copyright (C) 2018, 2019, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
    3.0.0	Support Location field
    		Use DBKeyHandler
	3.1.0	Use jakarta implementation of JSON
*/

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class WildlifeLister implements IWildlifeLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<IWildlife> useWildlife = new DBKeyHandler<>("wildlifeId");

	private final DBKeyHandler<IWildlifeSpecies> useWildlifeSpecies = new DBKeyHandler<>("wildlifeSpeciesId");

	private final DBKeyHandler<ILocation> useLocation = new DBKeyHandler<>("locationId");

	private boolean useFromDate = false;
	private LocalDate fromDate;
	private boolean useToDate = false;
	private LocalDate toDate;

	private boolean useWhere = false;

	WildlifeLister() {}

	@Override
	public List<IWildlife> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<IWildlife> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.error("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("WildlifeLister: fetch(): no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.error("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("WildlifeLister: fetch(): no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (IWildlife vx : vals)
		{
			MySQLCache.cacheWildlife.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completeWildlife = true; }

		populateWildlifeSpecies(vals, fetchAll);

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from wildlife as d ");
        query.append("left join (select * from comment where ownerType = 'WI') as c ");
        query.append("on d.wildlifeId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.wildlifeSpeciesId, d.wildlifeId, c.date");
        return query.toString();
    }   //  buildSQL_MySQL()

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from wildlife as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'WI') as c ");
        query.append("on d.wildlifeId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.wildlifeSpeciesId, d.wildlifeId, c_date");
        return query.toString();
    }   //  buildSQL_hsqldb()

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.wildlifeId as d_wildlifeId, " +
                                                        "d.wildlifeSpeciesId as d_wildlifeSpeciesId, " +
														"d.locationId as d_locationId, " +
                                                        "d.date as d_date, " +
                                                        "d.variety as d_variety, " +
                                                        "d.created as d_created, " +
                                                        "d.lastUpdated as d_lastUpdated," +
                                                        " c.* from wildlife as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'WI') as c ");
        query.append("on d.wildlifeId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.wildlifeSpeciesId, d.wildlifeId, c_date");
        return query.toString();
    }   //  buildSQL_MSSQLServer()

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;
		if (useWildlife.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useWildlife.getSQLClause("d"));
			first = false;
			useWildlife.clear();
		}

		if (useWildlifeSpecies.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useWildlifeSpecies.getSQLClause("d"));
			first = false;
			useWildlifeSpecies.clear();
		}

		if (useLocation.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useLocation.getSQLClause("d"));
			first = false;
			useLocation.clear();
		}

		if (this.useFromDate)
        {
            if (first) query.append(" where ");
            else query.append(" and");

            query.append(" d.date >= '").append(this.fromDate).append("'");
            first = false;
            this.useFromDate = false;
        }
        if (this.useToDate)
        {
            if (first) query.append(" where ");
            else query.append(" and");

            query.append(" d.date <= '").append(this.toDate).append("'");
            first = false;
            this.useToDate = false;
        }
    }   //  buildCommonSQL()

	private List<IWildlife> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<IWildlife> tempList = new ArrayList<>();

		Wildlife item = null;

		while (rs.next())
		{
			int wildlifeId = rs.getInt("d.wildlifeId");
			int wildlifeSpeciesId = rs.getInt("d.wildlifeSpeciesId");
			int tmp_locationId = rs.getInt("d.locationId");
			Integer locationId = rs.wasNull() ? null : tmp_locationId;
			LocalDate date = rs.getDate("d.date").toLocalDate();
			String variety = rs.getString("d.variety");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("wildlifeId: {}, wildlifeSpeciesId: {}, locationId: {}, date: {}, variety: {}, lastUpdated: {}, created: {}",
                wildlifeId, wildlifeSpeciesId,locationId, date, variety, lastUpdated, created);
			if (item != null && wildlifeId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"WI",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new Wildlife(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new Wildlife(wildlifeId, wildlifeSpeciesId, locationId, date, variety, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						wildlifeId,
						"WI",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new Wildlife(wildlifeId, wildlifeSpeciesId, locationId, date, variety, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<IWildlife> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<IWildlife> tempList = new ArrayList<>();

		Wildlife item = null;

		while (rs.next())
		{
			int wildlifeId = rs.getInt("wildlife.wildlifeId");
			int wildlifeSpeciesId = rs.getInt("wildlife.wildlifeSpeciesId");
			int tmp_locationId = rs.getInt("wildlife.locationId");
			Integer locationId = rs.wasNull() ? null : tmp_locationId;
			LocalDate date = rs.getDate("wildlife.date").toLocalDate();
			String variety = rs.getString("wildlife.variety");
			LocalDateTime lastUpdated = rs.getTimestamp("wildlife.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("wildlife.created").toLocalDateTime();
LOGGER.debug("wildlifeId: {}, wildlifeSpeciesId: {}, locationId: {}, date: {}, variety: {}, lastUpdated: {}, created: {}",
                wildlifeId, wildlifeSpeciesId, locationId, date, variety, lastUpdated, created);
			if (item != null && wildlifeId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"WI",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Wildlife(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Wildlife(wildlifeId, wildlifeSpeciesId, locationId, date, variety, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						wildlifeId,
						"WI",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Wildlife(wildlifeId, wildlifeSpeciesId, locationId, date, variety, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<IWildlife> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<IWildlife> tempList = new ArrayList<>();

		Wildlife item = null;

		while (rs.next())
		{
			int wildlifeId = rs.getInt("d_wildlifeId");
			int wildlifeSpeciesId = rs.getInt("d_wildlifeSpeciesId");
			int tmp_locationId = rs.getInt("d_locationId");
			Integer locationId = rs.wasNull() ? null : tmp_locationId;
			LocalDate date = rs.getDate("d_date").toLocalDate();
			String variety = rs.getString("d_variety");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("wildlifeId: {}, wildlifeSpeciesId: {}, locationId: {}, date: {}, variety: {}, lastUpdated: {}, created: {}",
                wildlifeId, wildlifeSpeciesId, locationId, date, variety, lastUpdated, created);
			if (item != null && wildlifeId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"WI",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Wildlife(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Wildlife(wildlifeId, wildlifeSpeciesId, locationId, date, variety, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						wildlifeId,
						"WI",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Wildlife(wildlifeId, wildlifeSpeciesId, locationId, date, variety, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populateWildlifeSpecies(List<IWildlife> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: TD
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateWildlifeSpecies()");

		if (fetchAll)
		{
			new WildlifeSpeciesLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			mapToInt(c -> ((Wildlife)c).getWildlifeSpeciesId()).filter(c -> c>0).distinct().
			filter(c -> !MySQLCache.cacheWildlifeSpecies.containsKey(c)).
			toArray();

		if (keys.length > 0)
		{
			new WildlifeSpeciesLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cacheWildlife.clear();
		MySQLCache.completeWildlife = false;
	}

	/**
	*
	*Select only the Wildlife entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	IWildlifeLister id(int... vals)
	{
		useWildlife.id(vals);
//		useWhere = true;
		useWhere = useWhere || useWildlife.isUsed();
		return this;
	}

	@Override
	public IWildlifeLister wildlife(IWildlife... items)
	{
		useWildlife.item(items);
//		useWhere = true;
		useWhere = useWhere || useWildlife.isUsed();
		return this;
	}

	@Override
	public IWildlifeLister wildlife(List<IWildlife> items)
	{
		useWildlife.item(items);
//		useWhere = true;
		useWhere = useWhere || useWildlife.isUsed();
		return this;
	}

	@Override
	public IWildlifeLister wildlifeSpecies(IWildlifeSpecies... items)
	{
		useWildlifeSpecies.item(items);
//		useWhere = true;
		useWhere = useWhere || useWildlifeSpecies.isUsed();
		return this;
	}

	@Override
	public IWildlifeLister wildlifeSpecies(List<IWildlifeSpecies> items)
	{
		useWildlifeSpecies.item(items);
//		useWhere = true;
		useWhere = useWhere || useWildlifeSpecies.isUsed();
		return this;
	}

	@Override
	public IWildlifeLister location(ILocation... items)
	{
		useLocation.item(items);
//		useWhere = true;
		useWhere = useWhere || useLocation.isUsed();
		return this;
	}

	@Override
	public IWildlifeLister location(List<ILocation> items)
	{
		useLocation.item(items);
//		useWhere = true;
		useWhere = useWhere || useLocation.isUsed();
		return this;
	}

	@Override
	public IWildlifeLister fromDate(LocalDate item)
	{
		if (item == null) return this;
		this.fromDate = item;
		this.useFromDate = true;
		this.useWhere = true;
		return this;
	}

	@Override
	public IWildlifeLister toDate(LocalDate item)
	{
		if (item == null) return this;
		this.toDate = item;
		this.useToDate = true;
		this.useWhere = true;
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IWildlife ihc : MySQLCache.cacheWildlife.values())
		{
			Wildlife hc = (Wildlife)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "Wildlife");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "Wildlife.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
